/* Test trucate cast Decimal -> Binary overflow.

   Copyright (C) 2017 Free Software Foundation, Inc.

   This file is part of the Decimal Floating Point C Library.

   Author: Rogerio Alves <rogealve@br.ibm.com>

   The Decimal Floating Point C Library is free software; you can
   redistribute it and/or modify it under the terms of the GNU Lesser
   General Public License version 2.1.

   The Decimal Floating Point C Library is distributed in the hope that
   it will be useful, but WITHOUT ANY WARRANTY; without even the implied
   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
   the GNU Lesser General Public License version 2.1 for more details.

   You should have received a copy of the GNU Lesser General Public
   License version 2.1 along with the Decimal Floating Point C Library;
   if not, write to the Free Software Foundation, Inc., 59 Temple Place,
   Suite 330, Boston, MA 02111-1307 USA.

   Please see libdfp/COPYING.txt for more information.  */

#ifndef __STDC_WANT_DEC_FP__
#define __STDC_WANT_DEC_FP__
#endif

#include <float.h>
#include <stdio.h>
#include <fenv.h>
#include <math.h>

#define _WANT_VC 1
#include "scaffold.c"

/* TODO: Radix conversion in the denormal range will fail today with the
   implementations provided by libdfp today. */
#if defined _LIBDFP_HAS_IDEAL_RDX_CVT
#define XFAIL_CONVERSION 0
#else
#define XFAIL_CONVERSION 1
#endif

typedef struct
{
  int line;
  int round_mode;
  _Decimal128 set;
  long double expect;
} d128_type;

typedef struct
{
  int line;
  int round_mode;
  _Decimal64 set;
  double expect;
} d64_type;

typedef struct
{
  int line;
  int round_mode;
  _Decimal32 set;
  float expect;
} d32_type;

/* Decimal 128 to Binary Double 128.  */
static const d128_type d128to128[] = {
 {__LINE__, FE_TONEAREST, 1e-6176DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-6176DL, -0.0},
 {__LINE__, FE_TONEAREST, 1e-4967DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-4967DL, -0.0},
#if __LDBL_MANT_DIG__ == 113
 {__LINE__, FE_TONEAREST, 3.237587559719012555462219479113823e-4966DL, 0.0},
 {__LINE__, FE_TONEAREST, -3.237587559719012555462219479113823e-4966DL, -0.0},
 {__LINE__, FE_TONEAREST, 3.237587559719012555462219479113824e-4966DL,
  __LDBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -3.237587559719012555462219479113824e-4966DL,
  -__LDBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 7e-4966DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7e-4966DL, -__LDBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 13e-4966DL, 2 * __LDBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -13e-4966DL, -2 * __LDBL_DENORM_MIN__},
#endif
 {__LINE__, FE_TOWARDZERO, 1e-5000DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-5000DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1e-4967DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-4967DL, -0.0},
#if __LDBL_MANT_DIG__ == 113
 {__LINE__, FE_TOWARDZERO, 6.475175119438025e-4966DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -6.475175119438025e-4966DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 6.475175119438025110924438958227646e-4966DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -6.475175119438025110924438958227646e-4966DL, -0.0 },
 {__LINE__, FE_TOWARDZERO, 6.475175119438025110924438958227647e-4966DL,
  __LDBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -6.475175119438025110924438958227647e-4966DL,
  -__LDBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 6.475175119438026e-4966DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -6.475175119438026e-4966DL, -__LDBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 7e-4966DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -7e-4966DL, -__LDBL_DENORM_MIN__},
#endif
 {__LINE__, FE_DOWNWARD, 1e-5000DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-5000DL, -__LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-4967DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-4967DL, -__LDBL_DENORM_MIN__},
#if __LDBL_MANT_DIG__ == 113
 {__LINE__, FE_DOWNWARD, 6.475175119438025e-4966DL, 0.0},
 {__LINE__, FE_DOWNWARD, -6.475175119438025e-4966DL, -__LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 6.475175119438025110924438958227646e-4966DL, 0.0},
 {__LINE__, FE_DOWNWARD, -6.475175119438025110924438958227646e-4966DL,
  -__LDBL_DENORM_MIN__ },
 {__LINE__, FE_DOWNWARD, 6.475175119438025110924438958227647e-4966DL,
  __LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -6.475175119438025110924438958227647e-4966DL,
  -2 * __LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 6.475175119438026e-4966DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -6.475175119438026e-4966DL, -2 * __LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 7e-4966DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -7e-4966DL, -2 * __LDBL_DENORM_MIN__},
#endif
 {__LINE__, FE_UPWARD, 1e-5000DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-5000DL, -0.0},
 {__LINE__, FE_UPWARD, 1e-4967DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-4967DL, -0.0},
#if __LDBL_MANT_DIG__ == 113
 {__LINE__, FE_UPWARD, 6.475175119438025e-4966DL, __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -6.475175119438025e-4966DL, -0.0},
 {__LINE__, FE_UPWARD, 6.475175119438025110924438958227646e-4966DL,
  __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -6.475175119438025110924438958227646e-4966DL, -0.0 },
 {__LINE__, FE_UPWARD, 6.475175119438025110924438958227647e-4966DL,
  2 * __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -6.475175119438025110924438958227647e-4966DL,
  -__LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 6.475175119438026e-4966DL, 2 * __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -6.475175119438026e-4966DL, -__LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 7e-4966DL, 2 * __LDBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -7e-4966DL, -__LDBL_DENORM_MIN__}
#endif
};

/* Decimal 128 to Binary Double 64.  */
static const d128_type d128to64[] = {
 {__LINE__, FE_TONEAREST, 1e-6176DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-6176DL, -0.0},
 {__LINE__, FE_TONEAREST, 1e-1000DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-1000DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TONEAREST, 1e-325DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-325DL, -0.0},
#endif
 {__LINE__, FE_TONEAREST, 2.4703282292062326e-324DL, 0.0},
 {__LINE__, FE_TONEAREST, -2.4703282292062326e-324DL, -0.0},
 {__LINE__, FE_TONEAREST, 2.4703282292062327e-324DL, 0.0},
 {__LINE__, FE_TONEAREST, -2.4703282292062327e-324DL, -0.0},
 {__LINE__, FE_TONEAREST, 2.470328229206232720882843964341105e-324DL, 0.0},
 {__LINE__, FE_TONEAREST, -2.470328229206232720882843964341105e-324DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TONEAREST, 2.470328229206232720882843964341107e-324DL,
  __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -2.470328229206232720882843964341107e-324DL,
  -__DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 2.4703282292062328e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -2.4703282292062328e-324DL, -__DBL_DENORM_MIN__},
#endif
 {__LINE__, FE_TONEAREST, 5e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -5e-324DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 1e-323DL, 2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -1e-323DL, -2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 1e-1000DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-1000DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1e-325DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-325DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 4.940656458412465e-324DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -4.940656458412465e-324DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 4.940656458412465441765687928682213e-324DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -4.940656458412465441765687928682213e-324DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TOWARDZERO, 4.940656458412465441765687928682214e-324DL,
  __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -4.940656458412465441765687928682214e-324DL,
  -__DBL_DENORM_MIN__},
#endif
 {__LINE__, FE_TOWARDZERO, 4.940656458412466e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -4.940656458412466e-324DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 5e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -5e-324DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-1000DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-1000DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-325DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-325DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 4.940656458412465e-324DL, 0.0},
 {__LINE__, FE_DOWNWARD, -4.940656458412465e-324DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 4.940656458412465441765687928682213e-324DL, 0.0},
 {__LINE__, FE_DOWNWARD, -4.940656458412465441765687928682213e-324DL,
  -__DBL_DENORM_MIN__},
#if !XFAIL_CONVERSION
 {__LINE__, FE_DOWNWARD, 4.940656458412465441765687928682214e-324DL,
  __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -4.940656458412465441765687928682214e-324DL,
  -2 * __DBL_DENORM_MIN__},
#endif
 {__LINE__, FE_DOWNWARD, 4.940656458412466e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -4.940656458412466e-324DL, -2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 5e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -5e-324DL, -2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 1e-1000DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-1000DL, -0.0},
 {__LINE__, FE_UPWARD, 1e-325DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-325DL, -0.0},
 {__LINE__, FE_UPWARD, 4.940656458412465e-324DL, __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -4.940656458412465e-324DL, -0.0},
 {__LINE__, FE_UPWARD, 4.940656458412465441765687928682213e-324DL,
  __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -4.940656458412465441765687928682213e-324DL, -0.0 },
#if !XFAIL_CONVERSION
 {__LINE__, FE_UPWARD, 4.940656458412465441765687928682214e-324DL,
  2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -4.940656458412465441765687928682214e-324DL,
  -__DBL_DENORM_MIN__},
#endif
 {__LINE__, FE_UPWARD, 4.940656458412466e-324DL, 2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -4.940656458412466e-324DL, -__DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 5e-324DL, 2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -5e-324DL, -__DBL_DENORM_MIN__}
};

/* Decimal 128 to Binary Float 32.  */
static const d128_type d128to32[] = {
 {__LINE__, FE_TONEAREST, 1e-6176DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-6176DL, -0.0},
 {__LINE__, FE_TONEAREST, 1e-1000DL, 0.0},
 {__LINE__, FE_TONEAREST, -1e-1000DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TONEAREST, 7.0064922e-46DL, 0.0},
 {__LINE__, FE_TONEAREST, -7.0064922e-46DL, -0.0},
#endif
 {__LINE__, FE_TONEAREST, 7.0064923e-46DL, 0.0},
 {__LINE__, FE_TONEAREST, -7.0064923e-46DL, -0.0},
 {__LINE__, FE_TONEAREST, 7.006492321624085e-46DL, 0.0},
 {__LINE__, FE_TONEAREST, -7.006492321624085e-46DL, -0.0},
 {__LINE__, FE_TONEAREST, 7.006492321624085354618647916449580e-46DL, 0.0},
 {__LINE__, FE_TONEAREST, -7.006492321624085354618647916449580e-46DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TONEAREST, 7.006492321624085354618647916449581e-46DL,
  __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.006492321624085354618647916449581e-46DL,
  -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 7.006492321624086e-46DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.006492321624086e-46DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 7.0064924e-46DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.0064924e-46DL, -__FLT_DENORM_MIN__},
#endif
 {__LINE__, FE_TONEAREST, 1.5e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -1.5e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 2.9e-45DL, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -2.9e-45DL, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 1e-1000DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-1000DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1e-46DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-46DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1.401298464324817e-45DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1.401298464324817e-45DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1.401298464324817070923729583289916e-45DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1.401298464324817070923729583289916e-45DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TOWARDZERO, 1.401298464324817070923729583289917e-45DL,
  __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -1.401298464324817070923729583289917e-45DL,
  -__FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 1.401298464324818e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -1.401298464324818e-45DL, -__FLT_DENORM_MIN__},
#endif
 {__LINE__, FE_TOWARDZERO, 2e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -2e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-1000DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-1000DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-46DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-46DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401298464324817e-45DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1.401298464324817e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401298464324817070923729583289916e-45DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1.401298464324817070923729583289916e-45DL,
  -__FLT_DENORM_MIN__},
#if !XFAIL_CONVERSION
 {__LINE__, FE_DOWNWARD, 1.401298464324817070923729583289917e-45DL,
  __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -1.401298464324817070923729583289917e-45DL,
  -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401298464324818e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -1.401298464324818e-45DL, -2 * __FLT_DENORM_MIN__},
#endif
 {__LINE__, FE_DOWNWARD, 2e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -2e-45DL, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 1e-1000DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-1000DL, -0.0},
 {__LINE__, FE_UPWARD, 1e-46DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-46DL, -0.0},
 {__LINE__, FE_UPWARD, 1.401298464324817e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298464324817e-45DL, -0.0},
 {__LINE__, FE_UPWARD, 1.401298464324817070923729583289916e-45DL,
  __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298464324817070923729583289916e-45DL, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_UPWARD, 1.401298464324817070923729583289917e-45DL,
  2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298464324817070923729583289917e-45DL,
  -__FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 1.401298464324818e-45DL, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298464324818e-45DL, -__FLT_DENORM_MIN__},
#endif
 {__LINE__, FE_UPWARD, 2e-45DL, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -2e-45DL, -__FLT_DENORM_MIN__}
};

/* Decimal 64 to Binary Double 64.  */
static const d64_type d64to64[] = {
 {__LINE__, FE_TONEAREST, 1e-398DD, 0.0},
 {__LINE__, FE_TONEAREST, -1e-398DD, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TONEAREST, 1e-325DD, 0.0},
 {__LINE__, FE_TONEAREST, -1e-325DD, -0.0},
#endif
 {__LINE__, FE_TONEAREST, 2.470328229206232e-324DD, 0.0},
 {__LINE__, FE_TONEAREST, -2.470328229206232e-324DD, -0.0},
 {__LINE__, FE_TONEAREST, 2.470328229206233e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -2.470328229206233e-324DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 5e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -5e-324DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 1e-323DD, 2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -1e-323DD, -2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 1e-398DD, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-398DD, -0.0},
 {__LINE__, FE_TOWARDZERO, 1e-325DD, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-325DD, -0.0},
 {__LINE__, FE_TOWARDZERO, 4.940656458412465e-324DD, 0.0},
 {__LINE__, FE_TOWARDZERO, -4.940656458412465e-324DD, -0.0},
 {__LINE__, FE_TOWARDZERO, 4.940656458412466e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -4.940656458412466e-324DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 5e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -5e-324DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-398DD, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-398DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-325DD, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-325DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 4.940656458412465e-324DD, 0.0},
 {__LINE__, FE_DOWNWARD, -4.940656458412465e-324DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 4.940656458412466e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -4.940656458412466e-324DD, -2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 5e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -5e-324DD, -2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 1e-398DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-398DD, -0.0},
 {__LINE__, FE_UPWARD, 1e-325DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-325DD, -0.0},
 {__LINE__, FE_UPWARD, 4.940656458412465e-324DD, __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -4.940656458412465e-324DD, -0.0},
 {__LINE__, FE_UPWARD, 4.940656458412466e-324DD, 2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -4.940656458412466e-324DD, -__DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 5e-324DD, 2 * __DBL_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -5e-324DD, -__DBL_DENORM_MIN__}
};

/* Decimal 64 to Binary Float 32.  */
static const d64_type d64to32[] = {
 {__LINE__, FE_TONEAREST, 1e-398DD, 0.0},
 {__LINE__, FE_TONEAREST, -1e-398DD, -0.0},
 {__LINE__, FE_TONEAREST, 7.0064922e-46DD, 0.0},
 {__LINE__, FE_TONEAREST, -7.0064922e-46DD, -0.0},
 {__LINE__, FE_TONEAREST, 7.0064923e-46DD, 0.0},
 {__LINE__, FE_TONEAREST, -7.0064923e-46DD, -0.0},
 {__LINE__, FE_TONEAREST, 7.00649232e-46DD, 0.0},
 {__LINE__, FE_TONEAREST, -7.00649232e-46DD, -0.0},
 {__LINE__, FE_TONEAREST, 7.006492321624085e-46DD, 0.0},
 {__LINE__, FE_TONEAREST, -7.006492321624085e-46DD, -0.0},
#if !XFAIL_CONVERSION
 {__LINE__, FE_TONEAREST, 7.006492321624086e-46DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.006492321624086e-46DD, -__FLT_DENORM_MIN__},
#endif
 {__LINE__, FE_TONEAREST, 7.00649233e-46DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.00649233e-46DD, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 7.0064924e-46DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.0064924e-46DD, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 1.5e-45DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -1.5e-45DD, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 2.9e-45DD, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -2.9e-45DD, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 1e-398DD, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-398DD, -0.0},
 {__LINE__, FE_TOWARDZERO, 1e-46DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-46DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1.401298464324817e-45DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1.401298464324817e-45DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1.401298464324818e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -1.401298464324818e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 2e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -2e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-398DD, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-398DD, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-46DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-46DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401298464324817e-45DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1.401298464324817e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401298464324818e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -1.401298464324818e-45DL, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 2e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -2e-45DL, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 1e-398DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-398DD, -0.0},
 {__LINE__, FE_UPWARD, 1e-46DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-46DD, -0.0},
 {__LINE__, FE_UPWARD, 1.401298464324817e-45DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298464324817e-45DD, -0.0},
 {__LINE__, FE_UPWARD, 1.401298464324818e-45DD, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298464324818e-45DD, -__FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 2e-45DD, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -2e-45DD, -__FLT_DENORM_MIN__}
};

/* Decimal 32 to Binary Float 32.  */
static const d32_type d32to32[] = {
 {__LINE__, FE_TONEAREST, 1e-101DF, 0.0},
 {__LINE__, FE_TONEAREST, -1e-101DF, -0.0},
 {__LINE__, FE_TONEAREST, 7.006492e-46DF, 0.0},
 {__LINE__, FE_TONEAREST, -7.006492e-46DF, -0.0},
 {__LINE__, FE_TONEAREST, 7.006493e-46DF, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -7.006493e-46DF, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 1.5e-45DF, __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -1.5e-45DF, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, 2.9e-45DF, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_TONEAREST, -2.9e-45DF, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 1e-101DF, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-101DF, -0.0},
 {__LINE__, FE_TOWARDZERO, 1e-46DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1e-46DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1.401298e-45DL, 0.0},
 {__LINE__, FE_TOWARDZERO, -1.401298e-45DL, -0.0},
 {__LINE__, FE_TOWARDZERO, 1.401299e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -1.401299e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, 2e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_TOWARDZERO, -2e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-101DF, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-101DF, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1e-46DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1e-46DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401298e-45DL, 0.0},
 {__LINE__, FE_DOWNWARD, -1.401298e-45DL, -__FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 1.401299e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -1.401299e-45DL, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, 2e-45DL, __FLT_DENORM_MIN__},
 {__LINE__, FE_DOWNWARD, -2e-45DL, -2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 1e-101DF, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-101DF, -0.0},
 {__LINE__, FE_UPWARD, 1e-46DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1e-46DD, -0.0},
 {__LINE__, FE_UPWARD, 1.401298e-45DD, __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401298e-45DD, -0.0},
 {__LINE__, FE_UPWARD, 1.401299e-45DD, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -1.401299e-45DD, -__FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, 2e-45DD, 2 * __FLT_DENORM_MIN__},
 {__LINE__, FE_UPWARD, -2e-45DD, -__FLT_DENORM_MIN__}
};

static const int d128to128_size = sizeof (d128to128) / sizeof (d128to128[0]);
static const int d128to64_size = sizeof (d128to64) / sizeof (d128to64[0]);
static const int d128to32_size = sizeof (d128to32) / sizeof (d128to32[0]);

static const int d64to64_size = sizeof (d64to64) / sizeof (d64to64[0]);
static const int d64to32_size = sizeof (d64to32) / sizeof (d64to32[0]);

static const int d32to32_size = sizeof (d32to32) / sizeof (d32to32[0]);

int
main(void)
{
  int i;
  for (i=0; i<d128to128_size; ++i)
    {
      long double ret;
      fesetround (d128to128[i].round_mode);
      ret = (long double) d128to128[i].set;
      _VC_P (__FILE__, d128to128[i].line, d128to128[i].expect,
	     (long double) ret, "%La");
    }

  for (i=0; i<d128to64_size; ++i)
    {
      double ret;
      fesetround (d128to64[i].round_mode);
      ret = (double) d128to64[i].set;
      _VC_P (__FILE__, d128to64[i].line, d128to64[i].expect, (long double) ret,
	     "%La");
    }

  for (i=0; i<d128to32_size; ++i)
    {
      float ret;
      fesetround (d128to32[i].round_mode);
      ret = (float) d128to32[i].set;
      _VC_P (__FILE__, d128to32[i].line, d128to32[i].expect, (long double) ret,
	     "%La");
    }

  for (i=0; i<d64to64_size; ++i)
    {
      double ret;
      fesetround (d64to64[i].round_mode);
      ret = (double) d64to64[i].set;
      _VC_P (__FILE__, d64to64[i].line, d64to64[i].expect, ret, "%a");
    }

  for (i=0; i<d64to32_size; ++i)
    {
      float ret;
      fesetround (d64to32[i].round_mode);
      ret = (float) d64to32[i].set;
      _VC_P (__FILE__, d64to32[i].line, d64to32[i].expect, ret, "%a");
    }

  for (i=0; i<d32to32_size; ++i)
    {
      float ret;
      fesetround (d32to32[i].round_mode);
      ret = (float) d32to32[i].set;
      _VC_P (__FILE__, d32to32[i].line, d32to32[i].expect, ret, "%a");
    }

  _REPORT();

  /* fail comes from scaffold.c  */
  return fail;
}
