/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#if defined(__aarch64__) && (defined(FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC))

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_ffhybrid_fp16_mla_6x32 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<__fp16> A_arg,
    size_t M, size_t N, const __fp16 *B_ptr, size_t B_stride, IndirectOutputArg<__fp16> output_arg,
    const __fp16 *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
        __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const __fp16 *B_ptr = {};
        const __fp16 *cur_B_ptr = {};
        size_t B_stride = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const __fp16 *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    ka.B_stride = B_stride;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 251f\n"
      "cmp %x[M], #0x4\n"
      "bgt 201f\n"
      "beq 151f\n"
      "cmp %x[M], #0x2\n"
      "bgt 101f\n"
      "beq 51f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x14, #0x18\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 3f\n"
      "cmp x14, #0x10\n"
      "mov x9, x12\n"
      "bgt 3f\n"
      "cmp x14, #0x8\n"
      "mov x10, x12\n"
      "bgt 3f\n"
      "mov x11, x12\n"
      "3:"  // Height 1: B setup done
      "cbz x15, 4f\n"
      "ldr q8, [x15, #0x0]\n"
      "ldr q9, [x15, #0x10]\n"
      "ldr q10, [x15, #0x20]\n"
      "ldr q11, [x15, #0x30]\n"
      "add x15, x15, #0x40\n"
      "b 23f\n"
      "4:"  // Height 1: no bias
      "tbz %x[flags], #0, 22f\n"
      "cmp x14, #0x20\n"
      "bge 21f\n"
      "tbz x14, #4, 12f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v9.8h }, [x13], #0x10\n"
      "tbz x14, #3, 8f\n"
      "ld1 { v10.8h }, [x13], #0x10\n"
      "tbz x14, #2, 6f\n"
      "ldr d11, [x13], #0x8\n"
      "tbz x14, #1, 5f\n"
      "ld1 { v11.s }[2], [x13], #0x4\n"
      "mov x20, #0x3c\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v11.h }[6], [x13]\n"
      "b 20f\n"
      "5:"  // Height 1: Partial accumulate: partial_1_28
      "mov x20, #0x38\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v11.h }[4], [x13]\n"
      "b 20f\n"
      "6:"  // Height 1: Partial accumulate: partial_2_24
      "tbz x14, #1, 7f\n"
      "ldr s11, [x13], #0x4\n"
      "mov x20, #0x34\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v11.h }[2], [x13]\n"
      "b 20f\n"
      "7:"  // Height 1: Partial accumulate: partial_1_24
      "mov x20, #0x30\n"
      "tbz x14, #0, 20f\n"
      "ldr h11, [x13, #0x0]\n"
      "b 20f\n"
      "8:"  // Height 1: Partial accumulate: partial_4_16
      "tbz x14, #2, 10f\n"
      "ldr d10, [x13], #0x8\n"
      "tbz x14, #1, 9f\n"
      "ld1 { v10.s }[2], [x13], #0x4\n"
      "mov x20, #0x2c\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v10.h }[6], [x13]\n"
      "b 20f\n"
      "9:"  // Height 1: Partial accumulate: partial_1_20
      "mov x20, #0x28\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v10.h }[4], [x13]\n"
      "b 20f\n"
      "10:"  // Height 1: Partial accumulate: partial_2_16
      "tbz x14, #1, 11f\n"
      "ldr s10, [x13], #0x4\n"
      "mov x20, #0x24\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v10.h }[2], [x13]\n"
      "b 20f\n"
      "11:"  // Height 1: Partial accumulate: partial_1_16
      "mov x20, #0x20\n"
      "tbz x14, #0, 20f\n"
      "ldr h10, [x13, #0x0]\n"
      "b 20f\n"
      "12:"  // Height 1: Partial accumulate: partial_8_0
      "tbz x14, #3, 16f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "tbz x14, #2, 14f\n"
      "ldr d9, [x13], #0x8\n"
      "tbz x14, #1, 13f\n"
      "ld1 { v9.s }[2], [x13], #0x4\n"
      "mov x20, #0x1c\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v9.h }[6], [x13]\n"
      "b 20f\n"
      "13:"  // Height 1: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v9.h }[4], [x13]\n"
      "b 20f\n"
      "14:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x14, #1, 15f\n"
      "ldr s9, [x13], #0x4\n"
      "mov x20, #0x14\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v9.h }[2], [x13]\n"
      "b 20f\n"
      "15:"  // Height 1: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x14, #0, 20f\n"
      "ldr h9, [x13, #0x0]\n"
      "b 20f\n"
      "16:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x14, #2, 18f\n"
      "ldr d8, [x13], #0x8\n"
      "tbz x14, #1, 17f\n"
      "ld1 { v8.s }[2], [x13], #0x4\n"
      "mov x20, #0xc\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v8.h }[6], [x13]\n"
      "b 20f\n"
      "17:"  // Height 1: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v8.h }[4], [x13]\n"
      "b 20f\n"
      "18:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x14, #1, 19f\n"
      "ldr s8, [x13], #0x4\n"
      "mov x20, #0x4\n"
      "tbz x14, #0, 20f\n"
      "ld1 { v8.h }[2], [x13]\n"
      "b 20f\n"
      "19:"  // Height 1: Partial accumulate: partial_1_0
      "ldr h8, [x13, #0x0]\n"
      "mov x20, #0x0\n"
      "20:"  // Height 1: Partial accumulate: Done
      "sub x13, x13, x20\n"
      "b 23f\n"
      "21:"  // Height 1: full accumulate
      "ldr q8, [x13, #0x0]\n"
      "ldr q9, [x13, #0x10]\n"
      "ldr q10, [x13, #0x20]\n"
      "ldr q11, [x13, #0x30]\n"
      "b 23f\n"
      "22:"  // Height 1: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "23:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "24:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 25f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 26f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "b 26f\n"
      "25:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "26:"  // Height 1: input setup done
      "cmp x27, #0x8\n"
      "blt 29f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q6, [x12, #0x0]\n"
      "cmp x27, #0x10\n"
      "ldr q7, [x11, #0x0]\n"
      "blt 28f\n"
      "27:"  // Height 1: Multiply loop: Main loop head
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "ldr q17, [x10, #0x0]\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "ldr q16, [x9, #0x0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x10\n"
      "fmla v10.8h, v17.8h, v0.h[0]\n"
      "ldr q17, [x12, #0x10]\n"
      "fmla v11.8h, v16.8h, v0.h[0]\n"
      "ldr q16, [x11, #0x10]\n"
      "fmla v8.8h, v17.8h, v0.h[1]\n"
      "ldr q17, [x10, #0x10]\n"
      "fmla v9.8h, v16.8h, v0.h[1]\n"
      "ldr q16, [x9, #0x10]\n"
      "fmla v10.8h, v17.8h, v0.h[1]\n"
      "ldr q17, [x12, #0x20]\n"
      "fmla v11.8h, v16.8h, v0.h[1]\n"
      "ldr q16, [x11, #0x20]\n"
      "fmla v8.8h, v17.8h, v0.h[2]\n"
      "ldr q17, [x10, #0x20]\n"
      "fmla v9.8h, v16.8h, v0.h[2]\n"
      "ldr q16, [x9, #0x20]\n"
      "fmla v10.8h, v17.8h, v0.h[2]\n"
      "ldr q17, [x12, #0x30]\n"
      "fmla v11.8h, v16.8h, v0.h[2]\n"
      "ldr q16, [x11, #0x30]\n"
      "fmla v8.8h, v17.8h, v0.h[3]\n"
      "ldr q17, [x10, #0x30]\n"
      "fmla v9.8h, v16.8h, v0.h[3]\n"
      "ldr q16, [x9, #0x30]\n"
      "fmla v10.8h, v17.8h, v0.h[3]\n"
      "ldr q17, [x12, #0x40]\n"
      "fmla v11.8h, v16.8h, v0.h[3]\n"
      "ldr q16, [x11, #0x40]\n"
      "fmla v8.8h, v17.8h, v0.h[4]\n"
      "ldr q17, [x10, #0x40]\n"
      "fmla v9.8h, v16.8h, v0.h[4]\n"
      "ldr q16, [x9, #0x40]\n"
      "fmla v10.8h, v17.8h, v0.h[4]\n"
      "ldr q17, [x12, #0x50]\n"
      "fmla v11.8h, v16.8h, v0.h[4]\n"
      "ldr q16, [x11, #0x50]\n"
      "fmla v8.8h, v17.8h, v0.h[5]\n"
      "ldr q17, [x10, #0x50]\n"
      "fmla v9.8h, v16.8h, v0.h[5]\n"
      "ldr q16, [x9, #0x50]\n"
      "fmla v10.8h, v17.8h, v0.h[5]\n"
      "ldr q17, [x12, #0x60]\n"
      "fmla v11.8h, v16.8h, v0.h[5]\n"
      "ldr q16, [x11, #0x60]\n"
      "fmla v8.8h, v17.8h, v0.h[6]\n"
      "ldr q17, [x10, #0x60]\n"
      "fmla v9.8h, v16.8h, v0.h[6]\n"
      "ldr q16, [x9, #0x60]\n"
      "fmla v10.8h, v17.8h, v0.h[6]\n"
      "ldr q17, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v16.8h, v0.h[6]\n"
      "ldr q16, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v17.8h, v0.h[7]\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v16.8h, v0.h[7]\n"
      "ldr q16, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v17.8h, v0.h[7]\n"
      "ldr q6, [x12, #0x0]\n"
      "fmla v11.8h, v16.8h, v0.h[7]\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "bge 27b\n"
      "28:"  // Height 1: Multiply loop: Single iteration only
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "ldr q17, [x10, #0x0]\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "ldr q16, [x9, #0x0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v10.8h, v17.8h, v0.h[0]\n"
      "ldr q17, [x12, #0x10]\n"
      "fmla v11.8h, v16.8h, v0.h[0]\n"
      "ldr q16, [x11, #0x10]\n"
      "fmla v8.8h, v17.8h, v0.h[1]\n"
      "ldr q17, [x10, #0x10]\n"
      "fmla v9.8h, v16.8h, v0.h[1]\n"
      "ldr q16, [x9, #0x10]\n"
      "fmla v10.8h, v17.8h, v0.h[1]\n"
      "ldr q17, [x12, #0x20]\n"
      "fmla v11.8h, v16.8h, v0.h[1]\n"
      "ldr q16, [x11, #0x20]\n"
      "fmla v8.8h, v17.8h, v0.h[2]\n"
      "ldr q17, [x10, #0x20]\n"
      "fmla v9.8h, v16.8h, v0.h[2]\n"
      "ldr q16, [x9, #0x20]\n"
      "fmla v10.8h, v17.8h, v0.h[2]\n"
      "ldr q17, [x12, #0x30]\n"
      "fmla v11.8h, v16.8h, v0.h[2]\n"
      "ldr q16, [x11, #0x30]\n"
      "fmla v8.8h, v17.8h, v0.h[3]\n"
      "ldr q17, [x10, #0x30]\n"
      "fmla v9.8h, v16.8h, v0.h[3]\n"
      "ldr q16, [x9, #0x30]\n"
      "fmla v10.8h, v17.8h, v0.h[3]\n"
      "ldr q17, [x12, #0x40]\n"
      "fmla v11.8h, v16.8h, v0.h[3]\n"
      "ldr q16, [x11, #0x40]\n"
      "fmla v8.8h, v17.8h, v0.h[4]\n"
      "ldr q17, [x10, #0x40]\n"
      "fmla v9.8h, v16.8h, v0.h[4]\n"
      "ldr q16, [x9, #0x40]\n"
      "fmla v10.8h, v17.8h, v0.h[4]\n"
      "ldr q17, [x12, #0x50]\n"
      "fmla v11.8h, v16.8h, v0.h[4]\n"
      "ldr q16, [x11, #0x50]\n"
      "fmla v8.8h, v17.8h, v0.h[5]\n"
      "ldr q17, [x10, #0x50]\n"
      "fmla v9.8h, v16.8h, v0.h[5]\n"
      "ldr q16, [x9, #0x50]\n"
      "fmla v10.8h, v17.8h, v0.h[5]\n"
      "ldr q17, [x12, #0x60]\n"
      "fmla v11.8h, v16.8h, v0.h[5]\n"
      "ldr q16, [x11, #0x60]\n"
      "fmla v8.8h, v17.8h, v0.h[6]\n"
      "ldr q17, [x10, #0x60]\n"
      "fmla v9.8h, v16.8h, v0.h[6]\n"
      "ldr q16, [x9, #0x60]\n"
      "fmla v10.8h, v17.8h, v0.h[6]\n"
      "ldr q17, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v16.8h, v0.h[6]\n"
      "ldr q16, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v17.8h, v0.h[7]\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v16.8h, v0.h[7]\n"
      "ldr q16, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v17.8h, v0.h[7]\n"
      "fmla v11.8h, v16.8h, v0.h[7]\n"
      "29:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 31f\n"
      "30:"  // Height 1: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr q17, [x12, #0x0]\n"
      "sub x27, x27, #0x1\n"
      "add x12, x12, #0x10\n"
      "ldr q16, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "fmla v8.8h, v17.8h, v0.h[0]\n"
      "ldr q17, [x10, #0x0]\n"
      "add x10, x10, #0x10\n"
      "fmla v9.8h, v16.8h, v0.h[0]\n"
      "ldr q16, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "fmla v10.8h, v17.8h, v0.h[0]\n"
      "fmla v11.8h, v16.8h, v0.h[0]\n"
      "cbnz x27, 30b\n"
      "31:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 24b\n"
      "tbz %x[flags], #1, 32f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v17.8h }, [x21]\n"
      "ld1r { v16.8h }, [x20]\n"
      "fmin v8.8h, v8.8h, v17.8h\n"
      "fmin v9.8h, v9.8h, v17.8h\n"
      "fmin v10.8h, v10.8h, v17.8h\n"
      "fmin v11.8h, v11.8h, v17.8h\n"
      "fmax v8.8h, v8.8h, v16.8h\n"
      "fmax v9.8h, v9.8h, v16.8h\n"
      "fmax v10.8h, v10.8h, v16.8h\n"
      "fmax v11.8h, v11.8h, v16.8h\n"
      "32:"  // Height 1: No activation
      "cmp x14, #0x20\n"
      "bge 49f\n"
      "tbz x14, #4, 40f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v9.8h }, [x13], #0x10\n"
      "tbz x14, #3, 36f\n"
      "st1 { v10.8h }, [x13], #0x10\n"
      "tbz x14, #2, 34f\n"
      "str d11, [x13], #0x8\n"
      "tbz x14, #1, 33f\n"
      "st1 { v11.s }[2], [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v11.h }[6], [x13]\n"
      "b 48f\n"
      "33:"  // Height 1: Partial direct writeback: partial_1_28
      "tbz x14, #0, 48f\n"
      "st1 { v11.h }[4], [x13]\n"
      "b 48f\n"
      "34:"  // Height 1: Partial direct writeback: partial_2_24
      "tbz x14, #1, 35f\n"
      "str s11, [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v11.h }[2], [x13]\n"
      "b 48f\n"
      "35:"  // Height 1: Partial direct writeback: partial_1_24
      "tbz x14, #0, 48f\n"
      "str h11, [x13, #0x0]\n"
      "b 48f\n"
      "36:"  // Height 1: Partial direct writeback: partial_4_16
      "tbz x14, #2, 38f\n"
      "str d10, [x13], #0x8\n"
      "tbz x14, #1, 37f\n"
      "st1 { v10.s }[2], [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v10.h }[6], [x13]\n"
      "b 48f\n"
      "37:"  // Height 1: Partial direct writeback: partial_1_20
      "tbz x14, #0, 48f\n"
      "st1 { v10.h }[4], [x13]\n"
      "b 48f\n"
      "38:"  // Height 1: Partial direct writeback: partial_2_16
      "tbz x14, #1, 39f\n"
      "str s10, [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v10.h }[2], [x13]\n"
      "b 48f\n"
      "39:"  // Height 1: Partial direct writeback: partial_1_16
      "tbz x14, #0, 48f\n"
      "str h10, [x13, #0x0]\n"
      "b 48f\n"
      "40:"  // Height 1: Partial direct writeback: partial_8_0
      "tbz x14, #3, 44f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "tbz x14, #2, 42f\n"
      "str d9, [x13], #0x8\n"
      "tbz x14, #1, 41f\n"
      "st1 { v9.s }[2], [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v9.h }[6], [x13]\n"
      "b 48f\n"
      "41:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x14, #0, 48f\n"
      "st1 { v9.h }[4], [x13]\n"
      "b 48f\n"
      "42:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x14, #1, 43f\n"
      "str s9, [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v9.h }[2], [x13]\n"
      "b 48f\n"
      "43:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x14, #0, 48f\n"
      "str h9, [x13, #0x0]\n"
      "b 48f\n"
      "44:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x14, #2, 46f\n"
      "str d8, [x13], #0x8\n"
      "tbz x14, #1, 45f\n"
      "st1 { v8.s }[2], [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v8.h }[6], [x13]\n"
      "b 48f\n"
      "45:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x14, #0, 48f\n"
      "st1 { v8.h }[4], [x13]\n"
      "b 48f\n"
      "46:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x14, #1, 47f\n"
      "str s8, [x13], #0x4\n"
      "tbz x14, #0, 48f\n"
      "st1 { v8.h }[2], [x13]\n"
      "b 48f\n"
      "47:"  // Height 1: Partial direct writeback: partial_1_0
      "str h8, [x13, #0x0]\n"
      "48:"  // Height 1: Partial direct writeback: Done
      "b 50f\n"
      "49:"  // Height 1: Full writeback
      "str q8, [x13, #0x0]\n"
      "str q9, [x13, #0x10]\n"
      "str q10, [x13, #0x20]\n"
      "str q11, [x13, #0x30]\n"
      "add x13, x13, #0x40\n"
      "50:"  // Height 1: Writeback done
      "subs x14, x14, #0x20\n"
      "bgt 2b\n"
      "b 302f\n"
      "51:"  // Height 2
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "52:"  // Height 2: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x14, #0x18\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 53f\n"
      "cmp x14, #0x10\n"
      "mov x9, x12\n"
      "bgt 53f\n"
      "cmp x14, #0x8\n"
      "mov x10, x12\n"
      "bgt 53f\n"
      "mov x11, x12\n"
      "53:"  // Height 2: B setup done
      "cbz x15, 54f\n"
      "ldr q8, [x15, #0x0]\n"
      "ldr q9, [x15, #0x10]\n"
      "ldr q10, [x15, #0x20]\n"
      "ldr q11, [x15, #0x30]\n"
      "add x15, x15, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "b 73f\n"
      "54:"  // Height 2: no bias
      "tbz %x[flags], #0, 72f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x14, #0x20\n"
      "add x26, x13, x20, LSL #1\n"
      "bge 71f\n"
      "tbz x14, #4, 62f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v9.8h }, [x13], #0x10\n"
      "ld1 { v13.8h }, [x26], #0x10\n"
      "tbz x14, #3, 58f\n"
      "ld1 { v10.8h }, [x13], #0x10\n"
      "ld1 { v14.8h }, [x26], #0x10\n"
      "tbz x14, #2, 56f\n"
      "ldr d11, [x13], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "tbz x14, #1, 55f\n"
      "ld1 { v11.s }[2], [x13], #0x4\n"
      "ld1 { v15.s }[2], [x26], #0x4\n"
      "mov x20, #0x3c\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v11.h }[6], [x13]\n"
      "ld1 { v15.h }[6], [x26]\n"
      "b 70f\n"
      "55:"  // Height 2: Partial accumulate: partial_1_28
      "mov x20, #0x38\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v11.h }[4], [x13]\n"
      "ld1 { v15.h }[4], [x26]\n"
      "b 70f\n"
      "56:"  // Height 2: Partial accumulate: partial_2_24
      "tbz x14, #1, 57f\n"
      "ldr s11, [x13], #0x4\n"
      "ldr s15, [x26], #0x4\n"
      "mov x20, #0x34\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v11.h }[2], [x13]\n"
      "ld1 { v15.h }[2], [x26]\n"
      "b 70f\n"
      "57:"  // Height 2: Partial accumulate: partial_1_24
      "mov x20, #0x30\n"
      "tbz x14, #0, 70f\n"
      "ldr h11, [x13, #0x0]\n"
      "ldr h15, [x26, #0x0]\n"
      "b 70f\n"
      "58:"  // Height 2: Partial accumulate: partial_4_16
      "tbz x14, #2, 60f\n"
      "ldr d10, [x13], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "tbz x14, #1, 59f\n"
      "ld1 { v10.s }[2], [x13], #0x4\n"
      "ld1 { v14.s }[2], [x26], #0x4\n"
      "mov x20, #0x2c\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v10.h }[6], [x13]\n"
      "ld1 { v14.h }[6], [x26]\n"
      "b 70f\n"
      "59:"  // Height 2: Partial accumulate: partial_1_20
      "mov x20, #0x28\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v10.h }[4], [x13]\n"
      "ld1 { v14.h }[4], [x26]\n"
      "b 70f\n"
      "60:"  // Height 2: Partial accumulate: partial_2_16
      "tbz x14, #1, 61f\n"
      "ldr s10, [x13], #0x4\n"
      "ldr s14, [x26], #0x4\n"
      "mov x20, #0x24\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v10.h }[2], [x13]\n"
      "ld1 { v14.h }[2], [x26]\n"
      "b 70f\n"
      "61:"  // Height 2: Partial accumulate: partial_1_16
      "mov x20, #0x20\n"
      "tbz x14, #0, 70f\n"
      "ldr h10, [x13, #0x0]\n"
      "ldr h14, [x26, #0x0]\n"
      "b 70f\n"
      "62:"  // Height 2: Partial accumulate: partial_8_0
      "tbz x14, #3, 66f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "tbz x14, #2, 64f\n"
      "ldr d9, [x13], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "tbz x14, #1, 63f\n"
      "ld1 { v9.s }[2], [x13], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v9.h }[6], [x13]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "b 70f\n"
      "63:"  // Height 2: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v9.h }[4], [x13]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "b 70f\n"
      "64:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x14, #1, 65f\n"
      "ldr s9, [x13], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v9.h }[2], [x13]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "b 70f\n"
      "65:"  // Height 2: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x14, #0, 70f\n"
      "ldr h9, [x13, #0x0]\n"
      "ldr h13, [x26, #0x0]\n"
      "b 70f\n"
      "66:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x14, #2, 68f\n"
      "ldr d8, [x13], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "tbz x14, #1, 67f\n"
      "ld1 { v8.s }[2], [x13], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v8.h }[6], [x13]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "b 70f\n"
      "67:"  // Height 2: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v8.h }[4], [x13]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "b 70f\n"
      "68:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x14, #1, 69f\n"
      "ldr s8, [x13], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "tbz x14, #0, 70f\n"
      "ld1 { v8.h }[2], [x13]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "b 70f\n"
      "69:"  // Height 2: Partial accumulate: partial_1_0
      "ldr h8, [x13, #0x0]\n"
      "ldr h12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "70:"  // Height 2: Partial accumulate: Done
      "sub x13, x13, x20\n"
      "b 73f\n"
      "71:"  // Height 2: full accumulate
      "ldr q8, [x13, #0x0]\n"
      "ldr q9, [x13, #0x10]\n"
      "ldr q10, [x13, #0x20]\n"
      "ldr q11, [x13, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "b 73f\n"
      "72:"  // Height 2: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "73:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "74:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 75f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 76f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "b 76f\n"
      "75:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "76:"  // Height 2: input setup done
      "cmp x27, #0x8\n"
      "blt 79f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x10\n"
      "ldr q6, [x12, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "blt 78f\n"
      "77:"  // Height 2: Multiply loop: Main loop head
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "ldr q17, [x10, #0x0]\n"
      "sub x27, x27, #0x8\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "ldr q16, [x9, #0x0]\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v10.8h, v17.8h, v0.h[0]\n"
      "fmla v14.8h, v17.8h, v1.h[0]\n"
      "ldr q17, [x12, #0x10]\n"
      "fmla v11.8h, v16.8h, v0.h[0]\n"
      "fmla v15.8h, v16.8h, v1.h[0]\n"
      "ldr q16, [x11, #0x10]\n"
      "fmla v8.8h, v17.8h, v0.h[1]\n"
      "fmla v12.8h, v17.8h, v1.h[1]\n"
      "ldr q17, [x10, #0x10]\n"
      "fmla v9.8h, v16.8h, v0.h[1]\n"
      "fmla v13.8h, v16.8h, v1.h[1]\n"
      "ldr q16, [x9, #0x10]\n"
      "fmla v10.8h, v17.8h, v0.h[1]\n"
      "fmla v14.8h, v17.8h, v1.h[1]\n"
      "ldr q17, [x12, #0x20]\n"
      "fmla v11.8h, v16.8h, v0.h[1]\n"
      "fmla v15.8h, v16.8h, v1.h[1]\n"
      "ldr q16, [x11, #0x20]\n"
      "fmla v8.8h, v17.8h, v0.h[2]\n"
      "fmla v12.8h, v17.8h, v1.h[2]\n"
      "ldr q17, [x10, #0x20]\n"
      "fmla v9.8h, v16.8h, v0.h[2]\n"
      "fmla v13.8h, v16.8h, v1.h[2]\n"
      "ldr q16, [x9, #0x20]\n"
      "fmla v10.8h, v17.8h, v0.h[2]\n"
      "fmla v14.8h, v17.8h, v1.h[2]\n"
      "ldr q17, [x12, #0x30]\n"
      "fmla v11.8h, v16.8h, v0.h[2]\n"
      "fmla v15.8h, v16.8h, v1.h[2]\n"
      "ldr q16, [x11, #0x30]\n"
      "fmla v8.8h, v17.8h, v0.h[3]\n"
      "fmla v12.8h, v17.8h, v1.h[3]\n"
      "ldr q17, [x10, #0x30]\n"
      "fmla v9.8h, v16.8h, v0.h[3]\n"
      "fmla v13.8h, v16.8h, v1.h[3]\n"
      "ldr q16, [x9, #0x30]\n"
      "fmla v10.8h, v17.8h, v0.h[3]\n"
      "fmla v14.8h, v17.8h, v1.h[3]\n"
      "ldr q17, [x12, #0x40]\n"
      "fmla v11.8h, v16.8h, v0.h[3]\n"
      "fmla v15.8h, v16.8h, v1.h[3]\n"
      "ldr q16, [x11, #0x40]\n"
      "fmla v8.8h, v17.8h, v0.h[4]\n"
      "fmla v12.8h, v17.8h, v1.h[4]\n"
      "ldr q17, [x10, #0x40]\n"
      "fmla v9.8h, v16.8h, v0.h[4]\n"
      "fmla v13.8h, v16.8h, v1.h[4]\n"
      "ldr q16, [x9, #0x40]\n"
      "fmla v10.8h, v17.8h, v0.h[4]\n"
      "fmla v14.8h, v17.8h, v1.h[4]\n"
      "ldr q17, [x12, #0x50]\n"
      "fmla v11.8h, v16.8h, v0.h[4]\n"
      "fmla v15.8h, v16.8h, v1.h[4]\n"
      "ldr q16, [x11, #0x50]\n"
      "fmla v8.8h, v17.8h, v0.h[5]\n"
      "fmla v12.8h, v17.8h, v1.h[5]\n"
      "ldr q17, [x10, #0x50]\n"
      "fmla v9.8h, v16.8h, v0.h[5]\n"
      "fmla v13.8h, v16.8h, v1.h[5]\n"
      "ldr q16, [x9, #0x50]\n"
      "fmla v10.8h, v17.8h, v0.h[5]\n"
      "fmla v14.8h, v17.8h, v1.h[5]\n"
      "ldr q17, [x12, #0x60]\n"
      "fmla v11.8h, v16.8h, v0.h[5]\n"
      "fmla v15.8h, v16.8h, v1.h[5]\n"
      "ldr q16, [x11, #0x60]\n"
      "fmla v8.8h, v17.8h, v0.h[6]\n"
      "fmla v12.8h, v17.8h, v1.h[6]\n"
      "ldr q17, [x10, #0x60]\n"
      "fmla v9.8h, v16.8h, v0.h[6]\n"
      "fmla v13.8h, v16.8h, v1.h[6]\n"
      "ldr q16, [x9, #0x60]\n"
      "fmla v10.8h, v17.8h, v0.h[6]\n"
      "fmla v14.8h, v17.8h, v1.h[6]\n"
      "ldr q17, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v16.8h, v0.h[6]\n"
      "fmla v15.8h, v16.8h, v1.h[6]\n"
      "ldr q16, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v17.8h, v0.h[7]\n"
      "fmla v12.8h, v17.8h, v1.h[7]\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v16.8h, v0.h[7]\n"
      "fmla v13.8h, v16.8h, v1.h[7]\n"
      "ldr q16, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v17.8h, v0.h[7]\n"
      "fmla v14.8h, v17.8h, v1.h[7]\n"
      "ldr q6, [x12, #0x0]\n"
      "fmla v11.8h, v16.8h, v0.h[7]\n"
      "ldr q0, [x26, #0x0]\n"
      "fmla v15.8h, v16.8h, v1.h[7]\n"
      "ldr q1, [x25, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "bge 77b\n"
      "78:"  // Height 2: Multiply loop: Single iteration only
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "ldr q17, [x10, #0x0]\n"
      "sub x27, x27, #0x8\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "ldr q16, [x9, #0x0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v10.8h, v17.8h, v0.h[0]\n"
      "fmla v14.8h, v17.8h, v1.h[0]\n"
      "ldr q17, [x12, #0x10]\n"
      "fmla v11.8h, v16.8h, v0.h[0]\n"
      "fmla v15.8h, v16.8h, v1.h[0]\n"
      "ldr q16, [x11, #0x10]\n"
      "fmla v8.8h, v17.8h, v0.h[1]\n"
      "fmla v12.8h, v17.8h, v1.h[1]\n"
      "ldr q17, [x10, #0x10]\n"
      "fmla v9.8h, v16.8h, v0.h[1]\n"
      "fmla v13.8h, v16.8h, v1.h[1]\n"
      "ldr q16, [x9, #0x10]\n"
      "fmla v10.8h, v17.8h, v0.h[1]\n"
      "fmla v14.8h, v17.8h, v1.h[1]\n"
      "ldr q17, [x12, #0x20]\n"
      "fmla v11.8h, v16.8h, v0.h[1]\n"
      "fmla v15.8h, v16.8h, v1.h[1]\n"
      "ldr q16, [x11, #0x20]\n"
      "fmla v8.8h, v17.8h, v0.h[2]\n"
      "fmla v12.8h, v17.8h, v1.h[2]\n"
      "ldr q17, [x10, #0x20]\n"
      "fmla v9.8h, v16.8h, v0.h[2]\n"
      "fmla v13.8h, v16.8h, v1.h[2]\n"
      "ldr q16, [x9, #0x20]\n"
      "fmla v10.8h, v17.8h, v0.h[2]\n"
      "fmla v14.8h, v17.8h, v1.h[2]\n"
      "ldr q17, [x12, #0x30]\n"
      "fmla v11.8h, v16.8h, v0.h[2]\n"
      "fmla v15.8h, v16.8h, v1.h[2]\n"
      "ldr q16, [x11, #0x30]\n"
      "fmla v8.8h, v17.8h, v0.h[3]\n"
      "fmla v12.8h, v17.8h, v1.h[3]\n"
      "ldr q17, [x10, #0x30]\n"
      "fmla v9.8h, v16.8h, v0.h[3]\n"
      "fmla v13.8h, v16.8h, v1.h[3]\n"
      "ldr q16, [x9, #0x30]\n"
      "fmla v10.8h, v17.8h, v0.h[3]\n"
      "fmla v14.8h, v17.8h, v1.h[3]\n"
      "ldr q17, [x12, #0x40]\n"
      "fmla v11.8h, v16.8h, v0.h[3]\n"
      "fmla v15.8h, v16.8h, v1.h[3]\n"
      "ldr q16, [x11, #0x40]\n"
      "fmla v8.8h, v17.8h, v0.h[4]\n"
      "fmla v12.8h, v17.8h, v1.h[4]\n"
      "ldr q17, [x10, #0x40]\n"
      "fmla v9.8h, v16.8h, v0.h[4]\n"
      "fmla v13.8h, v16.8h, v1.h[4]\n"
      "ldr q16, [x9, #0x40]\n"
      "fmla v10.8h, v17.8h, v0.h[4]\n"
      "fmla v14.8h, v17.8h, v1.h[4]\n"
      "ldr q17, [x12, #0x50]\n"
      "fmla v11.8h, v16.8h, v0.h[4]\n"
      "fmla v15.8h, v16.8h, v1.h[4]\n"
      "ldr q16, [x11, #0x50]\n"
      "fmla v8.8h, v17.8h, v0.h[5]\n"
      "fmla v12.8h, v17.8h, v1.h[5]\n"
      "ldr q17, [x10, #0x50]\n"
      "fmla v9.8h, v16.8h, v0.h[5]\n"
      "fmla v13.8h, v16.8h, v1.h[5]\n"
      "ldr q16, [x9, #0x50]\n"
      "fmla v10.8h, v17.8h, v0.h[5]\n"
      "fmla v14.8h, v17.8h, v1.h[5]\n"
      "ldr q17, [x12, #0x60]\n"
      "fmla v11.8h, v16.8h, v0.h[5]\n"
      "fmla v15.8h, v16.8h, v1.h[5]\n"
      "ldr q16, [x11, #0x60]\n"
      "fmla v8.8h, v17.8h, v0.h[6]\n"
      "fmla v12.8h, v17.8h, v1.h[6]\n"
      "ldr q17, [x10, #0x60]\n"
      "fmla v9.8h, v16.8h, v0.h[6]\n"
      "fmla v13.8h, v16.8h, v1.h[6]\n"
      "ldr q16, [x9, #0x60]\n"
      "fmla v10.8h, v17.8h, v0.h[6]\n"
      "fmla v14.8h, v17.8h, v1.h[6]\n"
      "ldr q17, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v16.8h, v0.h[6]\n"
      "fmla v15.8h, v16.8h, v1.h[6]\n"
      "ldr q16, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v17.8h, v0.h[7]\n"
      "fmla v12.8h, v17.8h, v1.h[7]\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v16.8h, v0.h[7]\n"
      "fmla v13.8h, v16.8h, v1.h[7]\n"
      "ldr q16, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v17.8h, v0.h[7]\n"
      "fmla v14.8h, v17.8h, v1.h[7]\n"
      "fmla v11.8h, v16.8h, v0.h[7]\n"
      "fmla v15.8h, v16.8h, v1.h[7]\n"
      "79:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 81f\n"
      "80:"  // Height 2: Multiply loop: Odd block loop
      "ldr h1, [x26], #0x2\n"
      "ldr h0, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr q17, [x12, #0x0]\n"
      "ldr q16, [x11, #0x0]\n"
      "add x12, x12, #0x10\n"
      "add x11, x11, #0x10\n"
      "fmla v8.8h, v17.8h, v1.h[0]\n"
      "fmla v12.8h, v17.8h, v0.h[0]\n"
      "ldr q17, [x10, #0x0]\n"
      "add x10, x10, #0x10\n"
      "fmla v9.8h, v16.8h, v1.h[0]\n"
      "fmla v13.8h, v16.8h, v0.h[0]\n"
      "ldr q16, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "fmla v10.8h, v17.8h, v1.h[0]\n"
      "fmla v14.8h, v17.8h, v0.h[0]\n"
      "fmla v11.8h, v16.8h, v1.h[0]\n"
      "fmla v15.8h, v16.8h, v0.h[0]\n"
      "cbnz x27, 80b\n"
      "81:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 74b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "tbz %x[flags], #1, 82f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v17.8h }, [x21]\n"
      "ld1r { v16.8h }, [x20]\n"
      "fmin v8.8h, v8.8h, v17.8h\n"
      "fmin v9.8h, v9.8h, v17.8h\n"
      "fmin v10.8h, v10.8h, v17.8h\n"
      "fmin v11.8h, v11.8h, v17.8h\n"
      "fmin v12.8h, v12.8h, v17.8h\n"
      "fmin v13.8h, v13.8h, v17.8h\n"
      "fmin v14.8h, v14.8h, v17.8h\n"
      "fmin v15.8h, v15.8h, v17.8h\n"
      "fmax v8.8h, v8.8h, v16.8h\n"
      "fmax v9.8h, v9.8h, v16.8h\n"
      "fmax v10.8h, v10.8h, v16.8h\n"
      "fmax v11.8h, v11.8h, v16.8h\n"
      "fmax v12.8h, v12.8h, v16.8h\n"
      "fmax v13.8h, v13.8h, v16.8h\n"
      "fmax v14.8h, v14.8h, v16.8h\n"
      "fmax v15.8h, v15.8h, v16.8h\n"
      "82:"  // Height 2: No activation
      "cmp x14, #0x20\n"
      "bge 99f\n"
      "tbz x14, #4, 90f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v9.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v13.8h }, [x26], #0x10\n"
      "tbz x14, #3, 86f\n"
      "st1 { v10.8h }, [x13], #0x10\n"
      "st1 { v14.8h }, [x26], #0x10\n"
      "tbz x14, #2, 84f\n"
      "str d11, [x13], #0x8\n"
      "str d15, [x26], #0x8\n"
      "tbz x14, #1, 83f\n"
      "st1 { v11.s }[2], [x13], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v11.h }[6], [x13]\n"
      "st1 { v15.h }[6], [x26]\n"
      "b 98f\n"
      "83:"  // Height 2: Partial direct writeback: partial_1_28
      "tbz x14, #0, 98f\n"
      "st1 { v11.h }[4], [x13]\n"
      "st1 { v15.h }[4], [x26]\n"
      "b 98f\n"
      "84:"  // Height 2: Partial direct writeback: partial_2_24
      "tbz x14, #1, 85f\n"
      "str s11, [x13], #0x4\n"
      "str s15, [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v11.h }[2], [x13]\n"
      "st1 { v15.h }[2], [x26]\n"
      "b 98f\n"
      "85:"  // Height 2: Partial direct writeback: partial_1_24
      "tbz x14, #0, 98f\n"
      "str h11, [x13, #0x0]\n"
      "str h15, [x26, #0x0]\n"
      "b 98f\n"
      "86:"  // Height 2: Partial direct writeback: partial_4_16
      "tbz x14, #2, 88f\n"
      "str d10, [x13], #0x8\n"
      "str d14, [x26], #0x8\n"
      "tbz x14, #1, 87f\n"
      "st1 { v10.s }[2], [x13], #0x4\n"
      "st1 { v14.s }[2], [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v10.h }[6], [x13]\n"
      "st1 { v14.h }[6], [x26]\n"
      "b 98f\n"
      "87:"  // Height 2: Partial direct writeback: partial_1_20
      "tbz x14, #0, 98f\n"
      "st1 { v10.h }[4], [x13]\n"
      "st1 { v14.h }[4], [x26]\n"
      "b 98f\n"
      "88:"  // Height 2: Partial direct writeback: partial_2_16
      "tbz x14, #1, 89f\n"
      "str s10, [x13], #0x4\n"
      "str s14, [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v10.h }[2], [x13]\n"
      "st1 { v14.h }[2], [x26]\n"
      "b 98f\n"
      "89:"  // Height 2: Partial direct writeback: partial_1_16
      "tbz x14, #0, 98f\n"
      "str h10, [x13, #0x0]\n"
      "str h14, [x26, #0x0]\n"
      "b 98f\n"
      "90:"  // Height 2: Partial direct writeback: partial_8_0
      "tbz x14, #3, 94f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "tbz x14, #2, 92f\n"
      "str d9, [x13], #0x8\n"
      "str d13, [x26], #0x8\n"
      "tbz x14, #1, 91f\n"
      "st1 { v9.s }[2], [x13], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v9.h }[6], [x13]\n"
      "st1 { v13.h }[6], [x26]\n"
      "b 98f\n"
      "91:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x14, #0, 98f\n"
      "st1 { v9.h }[4], [x13]\n"
      "st1 { v13.h }[4], [x26]\n"
      "b 98f\n"
      "92:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x14, #1, 93f\n"
      "str s9, [x13], #0x4\n"
      "str s13, [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v9.h }[2], [x13]\n"
      "st1 { v13.h }[2], [x26]\n"
      "b 98f\n"
      "93:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x14, #0, 98f\n"
      "str h9, [x13, #0x0]\n"
      "str h13, [x26, #0x0]\n"
      "b 98f\n"
      "94:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x14, #2, 96f\n"
      "str d8, [x13], #0x8\n"
      "str d12, [x26], #0x8\n"
      "tbz x14, #1, 95f\n"
      "st1 { v8.s }[2], [x13], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v8.h }[6], [x13]\n"
      "st1 { v12.h }[6], [x26]\n"
      "b 98f\n"
      "95:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x14, #0, 98f\n"
      "st1 { v8.h }[4], [x13]\n"
      "st1 { v12.h }[4], [x26]\n"
      "b 98f\n"
      "96:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x14, #1, 97f\n"
      "str s8, [x13], #0x4\n"
      "str s12, [x26], #0x4\n"
      "tbz x14, #0, 98f\n"
      "st1 { v8.h }[2], [x13]\n"
      "st1 { v12.h }[2], [x26]\n"
      "b 98f\n"
      "97:"  // Height 2: Partial direct writeback: partial_1_0
      "str h8, [x13, #0x0]\n"
      "str h12, [x26, #0x0]\n"
      "98:"  // Height 2: Partial direct writeback: Done
      "b 100f\n"
      "99:"  // Height 2: Full writeback
      "str q8, [x13, #0x0]\n"
      "str q9, [x13, #0x10]\n"
      "str q10, [x13, #0x20]\n"
      "str q11, [x13, #0x30]\n"
      "add x13, x13, #0x40\n"
      "str q12, [x26, #0x0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "100:"  // Height 2: Writeback done
      "subs x14, x14, #0x20\n"
      "bgt 52b\n"
      "b 302f\n"
      "101:"  // Height 3
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "102:"  // Height 3: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x14, #0x18\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 103f\n"
      "cmp x14, #0x10\n"
      "mov x9, x12\n"
      "bgt 103f\n"
      "cmp x14, #0x8\n"
      "mov x10, x12\n"
      "bgt 103f\n"
      "mov x11, x12\n"
      "103:"  // Height 3: B setup done
      "cbz x15, 104f\n"
      "ldr q8, [x15, #0x0]\n"
      "ldr q9, [x15, #0x10]\n"
      "ldr q10, [x15, #0x20]\n"
      "ldr q11, [x15, #0x30]\n"
      "add x15, x15, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "b 123f\n"
      "104:"  // Height 3: no bias
      "tbz %x[flags], #0, 122f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x14, #0x20\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "bge 121f\n"
      "tbz x14, #4, 112f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v9.8h }, [x13], #0x10\n"
      "ld1 { v13.8h }, [x26], #0x10\n"
      "ld1 { v17.8h }, [x25], #0x10\n"
      "tbz x14, #3, 108f\n"
      "ld1 { v10.8h }, [x13], #0x10\n"
      "ld1 { v14.8h }, [x26], #0x10\n"
      "ld1 { v18.8h }, [x25], #0x10\n"
      "tbz x14, #2, 106f\n"
      "ldr d11, [x13], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "ldr d19, [x25], #0x8\n"
      "tbz x14, #1, 105f\n"
      "ld1 { v11.s }[2], [x13], #0x4\n"
      "ld1 { v15.s }[2], [x26], #0x4\n"
      "mov x20, #0x3c\n"
      "ld1 { v19.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v11.h }[6], [x13]\n"
      "ld1 { v15.h }[6], [x26]\n"
      "ld1 { v19.h }[6], [x25]\n"
      "b 120f\n"
      "105:"  // Height 3: Partial accumulate: partial_1_28
      "mov x20, #0x38\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v11.h }[4], [x13]\n"
      "ld1 { v15.h }[4], [x26]\n"
      "ld1 { v19.h }[4], [x25]\n"
      "b 120f\n"
      "106:"  // Height 3: Partial accumulate: partial_2_24
      "tbz x14, #1, 107f\n"
      "ldr s11, [x13], #0x4\n"
      "ldr s15, [x26], #0x4\n"
      "mov x20, #0x34\n"
      "ldr s19, [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v11.h }[2], [x13]\n"
      "ld1 { v15.h }[2], [x26]\n"
      "ld1 { v19.h }[2], [x25]\n"
      "b 120f\n"
      "107:"  // Height 3: Partial accumulate: partial_1_24
      "mov x20, #0x30\n"
      "tbz x14, #0, 120f\n"
      "ldr h11, [x13, #0x0]\n"
      "ldr h15, [x26, #0x0]\n"
      "ldr h19, [x25, #0x0]\n"
      "b 120f\n"
      "108:"  // Height 3: Partial accumulate: partial_4_16
      "tbz x14, #2, 110f\n"
      "ldr d10, [x13], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "ldr d18, [x25], #0x8\n"
      "tbz x14, #1, 109f\n"
      "ld1 { v10.s }[2], [x13], #0x4\n"
      "ld1 { v14.s }[2], [x26], #0x4\n"
      "mov x20, #0x2c\n"
      "ld1 { v18.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v10.h }[6], [x13]\n"
      "ld1 { v14.h }[6], [x26]\n"
      "ld1 { v18.h }[6], [x25]\n"
      "b 120f\n"
      "109:"  // Height 3: Partial accumulate: partial_1_20
      "mov x20, #0x28\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v10.h }[4], [x13]\n"
      "ld1 { v14.h }[4], [x26]\n"
      "ld1 { v18.h }[4], [x25]\n"
      "b 120f\n"
      "110:"  // Height 3: Partial accumulate: partial_2_16
      "tbz x14, #1, 111f\n"
      "ldr s10, [x13], #0x4\n"
      "ldr s14, [x26], #0x4\n"
      "mov x20, #0x24\n"
      "ldr s18, [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v10.h }[2], [x13]\n"
      "ld1 { v14.h }[2], [x26]\n"
      "ld1 { v18.h }[2], [x25]\n"
      "b 120f\n"
      "111:"  // Height 3: Partial accumulate: partial_1_16
      "mov x20, #0x20\n"
      "tbz x14, #0, 120f\n"
      "ldr h10, [x13, #0x0]\n"
      "ldr h14, [x26, #0x0]\n"
      "ldr h18, [x25, #0x0]\n"
      "b 120f\n"
      "112:"  // Height 3: Partial accumulate: partial_8_0
      "tbz x14, #3, 116f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "tbz x14, #2, 114f\n"
      "ldr d9, [x13], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "tbz x14, #1, 113f\n"
      "ld1 { v9.s }[2], [x13], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v9.h }[6], [x13]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "b 120f\n"
      "113:"  // Height 3: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v9.h }[4], [x13]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "b 120f\n"
      "114:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x14, #1, 115f\n"
      "ldr s9, [x13], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v9.h }[2], [x13]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "b 120f\n"
      "115:"  // Height 3: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x14, #0, 120f\n"
      "ldr h9, [x13, #0x0]\n"
      "ldr h13, [x26, #0x0]\n"
      "ldr h17, [x25, #0x0]\n"
      "b 120f\n"
      "116:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x14, #2, 118f\n"
      "ldr d8, [x13], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "tbz x14, #1, 117f\n"
      "ld1 { v8.s }[2], [x13], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v8.h }[6], [x13]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "b 120f\n"
      "117:"  // Height 3: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v8.h }[4], [x13]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "b 120f\n"
      "118:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x14, #1, 119f\n"
      "ldr s8, [x13], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "tbz x14, #0, 120f\n"
      "ld1 { v8.h }[2], [x13]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "b 120f\n"
      "119:"  // Height 3: Partial accumulate: partial_1_0
      "ldr h8, [x13, #0x0]\n"
      "ldr h12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr h16, [x25, #0x0]\n"
      "120:"  // Height 3: Partial accumulate: Done
      "sub x13, x13, x20\n"
      "b 123f\n"
      "121:"  // Height 3: full accumulate
      "ldr q8, [x13, #0x0]\n"
      "ldr q9, [x13, #0x10]\n"
      "ldr q10, [x13, #0x20]\n"
      "ldr q11, [x13, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0x0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "b 123f\n"
      "122:"  // Height 3: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "123:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "124:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 125f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 126f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "b 126f\n"
      "125:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "126:"  // Height 3: input setup done
      "cmp x27, #0x8\n"
      "blt 129f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q6, [x12, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "blt 128f\n"
      "127:"  // Height 3: Multiply loop: Main loop head
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "ldr q21, [x10, #0x0]\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "cmp x27, #0x10\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "ldr q20, [x9, #0x0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v10.8h, v21.8h, v0.h[0]\n"
      "fmla v14.8h, v21.8h, v1.h[0]\n"
      "fmla v18.8h, v21.8h, v2.h[0]\n"
      "ldr q21, [x12, #0x10]\n"
      "fmla v11.8h, v20.8h, v0.h[0]\n"
      "fmla v15.8h, v20.8h, v1.h[0]\n"
      "fmla v19.8h, v20.8h, v2.h[0]\n"
      "ldr q20, [x11, #0x10]\n"
      "fmla v8.8h, v21.8h, v0.h[1]\n"
      "fmla v12.8h, v21.8h, v1.h[1]\n"
      "fmla v16.8h, v21.8h, v2.h[1]\n"
      "ldr q21, [x10, #0x10]\n"
      "fmla v9.8h, v20.8h, v0.h[1]\n"
      "fmla v13.8h, v20.8h, v1.h[1]\n"
      "fmla v17.8h, v20.8h, v2.h[1]\n"
      "ldr q20, [x9, #0x10]\n"
      "fmla v10.8h, v21.8h, v0.h[1]\n"
      "fmla v14.8h, v21.8h, v1.h[1]\n"
      "fmla v18.8h, v21.8h, v2.h[1]\n"
      "ldr q21, [x12, #0x20]\n"
      "fmla v11.8h, v20.8h, v0.h[1]\n"
      "fmla v15.8h, v20.8h, v1.h[1]\n"
      "fmla v19.8h, v20.8h, v2.h[1]\n"
      "ldr q20, [x11, #0x20]\n"
      "fmla v8.8h, v21.8h, v0.h[2]\n"
      "fmla v12.8h, v21.8h, v1.h[2]\n"
      "fmla v16.8h, v21.8h, v2.h[2]\n"
      "ldr q21, [x10, #0x20]\n"
      "fmla v9.8h, v20.8h, v0.h[2]\n"
      "fmla v13.8h, v20.8h, v1.h[2]\n"
      "fmla v17.8h, v20.8h, v2.h[2]\n"
      "ldr q20, [x9, #0x20]\n"
      "fmla v10.8h, v21.8h, v0.h[2]\n"
      "fmla v14.8h, v21.8h, v1.h[2]\n"
      "fmla v18.8h, v21.8h, v2.h[2]\n"
      "ldr q21, [x12, #0x30]\n"
      "fmla v11.8h, v20.8h, v0.h[2]\n"
      "fmla v15.8h, v20.8h, v1.h[2]\n"
      "fmla v19.8h, v20.8h, v2.h[2]\n"
      "ldr q20, [x11, #0x30]\n"
      "fmla v8.8h, v21.8h, v0.h[3]\n"
      "fmla v12.8h, v21.8h, v1.h[3]\n"
      "fmla v16.8h, v21.8h, v2.h[3]\n"
      "ldr q21, [x10, #0x30]\n"
      "fmla v9.8h, v20.8h, v0.h[3]\n"
      "fmla v13.8h, v20.8h, v1.h[3]\n"
      "fmla v17.8h, v20.8h, v2.h[3]\n"
      "ldr q20, [x9, #0x30]\n"
      "fmla v10.8h, v21.8h, v0.h[3]\n"
      "fmla v14.8h, v21.8h, v1.h[3]\n"
      "fmla v18.8h, v21.8h, v2.h[3]\n"
      "ldr q21, [x12, #0x40]\n"
      "fmla v11.8h, v20.8h, v0.h[3]\n"
      "fmla v15.8h, v20.8h, v1.h[3]\n"
      "fmla v19.8h, v20.8h, v2.h[3]\n"
      "ldr q20, [x11, #0x40]\n"
      "fmla v8.8h, v21.8h, v0.h[4]\n"
      "fmla v12.8h, v21.8h, v1.h[4]\n"
      "fmla v16.8h, v21.8h, v2.h[4]\n"
      "ldr q21, [x10, #0x40]\n"
      "fmla v9.8h, v20.8h, v0.h[4]\n"
      "fmla v13.8h, v20.8h, v1.h[4]\n"
      "fmla v17.8h, v20.8h, v2.h[4]\n"
      "ldr q20, [x9, #0x40]\n"
      "fmla v10.8h, v21.8h, v0.h[4]\n"
      "fmla v14.8h, v21.8h, v1.h[4]\n"
      "fmla v18.8h, v21.8h, v2.h[4]\n"
      "ldr q21, [x12, #0x50]\n"
      "fmla v11.8h, v20.8h, v0.h[4]\n"
      "fmla v15.8h, v20.8h, v1.h[4]\n"
      "fmla v19.8h, v20.8h, v2.h[4]\n"
      "ldr q20, [x11, #0x50]\n"
      "fmla v8.8h, v21.8h, v0.h[5]\n"
      "fmla v12.8h, v21.8h, v1.h[5]\n"
      "fmla v16.8h, v21.8h, v2.h[5]\n"
      "ldr q21, [x10, #0x50]\n"
      "fmla v9.8h, v20.8h, v0.h[5]\n"
      "fmla v13.8h, v20.8h, v1.h[5]\n"
      "fmla v17.8h, v20.8h, v2.h[5]\n"
      "ldr q20, [x9, #0x50]\n"
      "fmla v10.8h, v21.8h, v0.h[5]\n"
      "fmla v14.8h, v21.8h, v1.h[5]\n"
      "fmla v18.8h, v21.8h, v2.h[5]\n"
      "ldr q21, [x12, #0x60]\n"
      "fmla v11.8h, v20.8h, v0.h[5]\n"
      "fmla v15.8h, v20.8h, v1.h[5]\n"
      "fmla v19.8h, v20.8h, v2.h[5]\n"
      "ldr q20, [x11, #0x60]\n"
      "fmla v8.8h, v21.8h, v0.h[6]\n"
      "fmla v12.8h, v21.8h, v1.h[6]\n"
      "fmla v16.8h, v21.8h, v2.h[6]\n"
      "ldr q21, [x10, #0x60]\n"
      "fmla v9.8h, v20.8h, v0.h[6]\n"
      "fmla v13.8h, v20.8h, v1.h[6]\n"
      "fmla v17.8h, v20.8h, v2.h[6]\n"
      "ldr q20, [x9, #0x60]\n"
      "fmla v10.8h, v21.8h, v0.h[6]\n"
      "fmla v14.8h, v21.8h, v1.h[6]\n"
      "fmla v18.8h, v21.8h, v2.h[6]\n"
      "ldr q21, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v20.8h, v0.h[6]\n"
      "fmla v15.8h, v20.8h, v1.h[6]\n"
      "fmla v19.8h, v20.8h, v2.h[6]\n"
      "ldr q20, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v21.8h, v0.h[7]\n"
      "fmla v12.8h, v21.8h, v1.h[7]\n"
      "fmla v16.8h, v21.8h, v2.h[7]\n"
      "ldr q21, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v20.8h, v0.h[7]\n"
      "fmla v13.8h, v20.8h, v1.h[7]\n"
      "fmla v17.8h, v20.8h, v2.h[7]\n"
      "ldr q20, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v21.8h, v0.h[7]\n"
      "fmla v14.8h, v21.8h, v1.h[7]\n"
      "fmla v18.8h, v21.8h, v2.h[7]\n"
      "ldr q6, [x12, #0x0]\n"
      "fmla v11.8h, v20.8h, v0.h[7]\n"
      "ldr q0, [x26, #0x0]\n"
      "fmla v15.8h, v20.8h, v1.h[7]\n"
      "ldr q1, [x25, #0x0]\n"
      "fmla v19.8h, v20.8h, v2.h[7]\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "bge 127b\n"
      "128:"  // Height 3: Multiply loop: Single iteration only
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "ldr q21, [x10, #0x0]\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "add x25, x25, #0x10\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "ldr q20, [x9, #0x0]\n"
      "add x24, x24, #0x10\n"
      "fmla v10.8h, v21.8h, v0.h[0]\n"
      "fmla v14.8h, v21.8h, v1.h[0]\n"
      "fmla v18.8h, v21.8h, v2.h[0]\n"
      "ldr q21, [x12, #0x10]\n"
      "fmla v11.8h, v20.8h, v0.h[0]\n"
      "fmla v15.8h, v20.8h, v1.h[0]\n"
      "fmla v19.8h, v20.8h, v2.h[0]\n"
      "ldr q20, [x11, #0x10]\n"
      "fmla v8.8h, v21.8h, v0.h[1]\n"
      "fmla v12.8h, v21.8h, v1.h[1]\n"
      "fmla v16.8h, v21.8h, v2.h[1]\n"
      "ldr q21, [x10, #0x10]\n"
      "fmla v9.8h, v20.8h, v0.h[1]\n"
      "fmla v13.8h, v20.8h, v1.h[1]\n"
      "fmla v17.8h, v20.8h, v2.h[1]\n"
      "ldr q20, [x9, #0x10]\n"
      "fmla v10.8h, v21.8h, v0.h[1]\n"
      "fmla v14.8h, v21.8h, v1.h[1]\n"
      "fmla v18.8h, v21.8h, v2.h[1]\n"
      "ldr q21, [x12, #0x20]\n"
      "fmla v11.8h, v20.8h, v0.h[1]\n"
      "fmla v15.8h, v20.8h, v1.h[1]\n"
      "fmla v19.8h, v20.8h, v2.h[1]\n"
      "ldr q20, [x11, #0x20]\n"
      "fmla v8.8h, v21.8h, v0.h[2]\n"
      "fmla v12.8h, v21.8h, v1.h[2]\n"
      "fmla v16.8h, v21.8h, v2.h[2]\n"
      "ldr q21, [x10, #0x20]\n"
      "fmla v9.8h, v20.8h, v0.h[2]\n"
      "fmla v13.8h, v20.8h, v1.h[2]\n"
      "fmla v17.8h, v20.8h, v2.h[2]\n"
      "ldr q20, [x9, #0x20]\n"
      "fmla v10.8h, v21.8h, v0.h[2]\n"
      "fmla v14.8h, v21.8h, v1.h[2]\n"
      "fmla v18.8h, v21.8h, v2.h[2]\n"
      "ldr q21, [x12, #0x30]\n"
      "fmla v11.8h, v20.8h, v0.h[2]\n"
      "fmla v15.8h, v20.8h, v1.h[2]\n"
      "fmla v19.8h, v20.8h, v2.h[2]\n"
      "ldr q20, [x11, #0x30]\n"
      "fmla v8.8h, v21.8h, v0.h[3]\n"
      "fmla v12.8h, v21.8h, v1.h[3]\n"
      "fmla v16.8h, v21.8h, v2.h[3]\n"
      "ldr q21, [x10, #0x30]\n"
      "fmla v9.8h, v20.8h, v0.h[3]\n"
      "fmla v13.8h, v20.8h, v1.h[3]\n"
      "fmla v17.8h, v20.8h, v2.h[3]\n"
      "ldr q20, [x9, #0x30]\n"
      "fmla v10.8h, v21.8h, v0.h[3]\n"
      "fmla v14.8h, v21.8h, v1.h[3]\n"
      "fmla v18.8h, v21.8h, v2.h[3]\n"
      "ldr q21, [x12, #0x40]\n"
      "fmla v11.8h, v20.8h, v0.h[3]\n"
      "fmla v15.8h, v20.8h, v1.h[3]\n"
      "fmla v19.8h, v20.8h, v2.h[3]\n"
      "ldr q20, [x11, #0x40]\n"
      "fmla v8.8h, v21.8h, v0.h[4]\n"
      "fmla v12.8h, v21.8h, v1.h[4]\n"
      "fmla v16.8h, v21.8h, v2.h[4]\n"
      "ldr q21, [x10, #0x40]\n"
      "fmla v9.8h, v20.8h, v0.h[4]\n"
      "fmla v13.8h, v20.8h, v1.h[4]\n"
      "fmla v17.8h, v20.8h, v2.h[4]\n"
      "ldr q20, [x9, #0x40]\n"
      "fmla v10.8h, v21.8h, v0.h[4]\n"
      "fmla v14.8h, v21.8h, v1.h[4]\n"
      "fmla v18.8h, v21.8h, v2.h[4]\n"
      "ldr q21, [x12, #0x50]\n"
      "fmla v11.8h, v20.8h, v0.h[4]\n"
      "fmla v15.8h, v20.8h, v1.h[4]\n"
      "fmla v19.8h, v20.8h, v2.h[4]\n"
      "ldr q20, [x11, #0x50]\n"
      "fmla v8.8h, v21.8h, v0.h[5]\n"
      "fmla v12.8h, v21.8h, v1.h[5]\n"
      "fmla v16.8h, v21.8h, v2.h[5]\n"
      "ldr q21, [x10, #0x50]\n"
      "fmla v9.8h, v20.8h, v0.h[5]\n"
      "fmla v13.8h, v20.8h, v1.h[5]\n"
      "fmla v17.8h, v20.8h, v2.h[5]\n"
      "ldr q20, [x9, #0x50]\n"
      "fmla v10.8h, v21.8h, v0.h[5]\n"
      "fmla v14.8h, v21.8h, v1.h[5]\n"
      "fmla v18.8h, v21.8h, v2.h[5]\n"
      "ldr q21, [x12, #0x60]\n"
      "fmla v11.8h, v20.8h, v0.h[5]\n"
      "fmla v15.8h, v20.8h, v1.h[5]\n"
      "fmla v19.8h, v20.8h, v2.h[5]\n"
      "ldr q20, [x11, #0x60]\n"
      "fmla v8.8h, v21.8h, v0.h[6]\n"
      "fmla v12.8h, v21.8h, v1.h[6]\n"
      "fmla v16.8h, v21.8h, v2.h[6]\n"
      "ldr q21, [x10, #0x60]\n"
      "fmla v9.8h, v20.8h, v0.h[6]\n"
      "fmla v13.8h, v20.8h, v1.h[6]\n"
      "fmla v17.8h, v20.8h, v2.h[6]\n"
      "ldr q20, [x9, #0x60]\n"
      "fmla v10.8h, v21.8h, v0.h[6]\n"
      "fmla v14.8h, v21.8h, v1.h[6]\n"
      "fmla v18.8h, v21.8h, v2.h[6]\n"
      "ldr q21, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v20.8h, v0.h[6]\n"
      "fmla v15.8h, v20.8h, v1.h[6]\n"
      "fmla v19.8h, v20.8h, v2.h[6]\n"
      "ldr q20, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v21.8h, v0.h[7]\n"
      "fmla v12.8h, v21.8h, v1.h[7]\n"
      "fmla v16.8h, v21.8h, v2.h[7]\n"
      "ldr q21, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v20.8h, v0.h[7]\n"
      "fmla v13.8h, v20.8h, v1.h[7]\n"
      "fmla v17.8h, v20.8h, v2.h[7]\n"
      "ldr q20, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v21.8h, v0.h[7]\n"
      "fmla v14.8h, v21.8h, v1.h[7]\n"
      "fmla v18.8h, v21.8h, v2.h[7]\n"
      "fmla v11.8h, v20.8h, v0.h[7]\n"
      "fmla v15.8h, v20.8h, v1.h[7]\n"
      "fmla v19.8h, v20.8h, v2.h[7]\n"
      "129:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 131f\n"
      "130:"  // Height 3: Multiply loop: Odd block loop
      "ldr h2, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h0, [x24], #0x2\n"
      "ldr q21, [x12, #0x0]\n"
      "add x12, x12, #0x10\n"
      "ldr q20, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "fmla v8.8h, v21.8h, v2.h[0]\n"
      "fmla v12.8h, v21.8h, v1.h[0]\n"
      "fmla v16.8h, v21.8h, v0.h[0]\n"
      "ldr q21, [x10, #0x0]\n"
      "add x10, x10, #0x10\n"
      "fmla v9.8h, v20.8h, v2.h[0]\n"
      "fmla v13.8h, v20.8h, v1.h[0]\n"
      "fmla v17.8h, v20.8h, v0.h[0]\n"
      "ldr q20, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "fmla v10.8h, v21.8h, v2.h[0]\n"
      "fmla v14.8h, v21.8h, v1.h[0]\n"
      "fmla v18.8h, v21.8h, v0.h[0]\n"
      "fmla v11.8h, v20.8h, v2.h[0]\n"
      "fmla v15.8h, v20.8h, v1.h[0]\n"
      "fmla v19.8h, v20.8h, v0.h[0]\n"
      "cbnz x27, 130b\n"
      "131:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 124b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "tbz %x[flags], #1, 132f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v21.8h }, [x21]\n"
      "ld1r { v20.8h }, [x20]\n"
      "fmin v8.8h, v8.8h, v21.8h\n"
      "fmin v9.8h, v9.8h, v21.8h\n"
      "fmin v10.8h, v10.8h, v21.8h\n"
      "fmin v11.8h, v11.8h, v21.8h\n"
      "fmin v12.8h, v12.8h, v21.8h\n"
      "fmin v13.8h, v13.8h, v21.8h\n"
      "fmin v14.8h, v14.8h, v21.8h\n"
      "fmin v15.8h, v15.8h, v21.8h\n"
      "fmin v16.8h, v16.8h, v21.8h\n"
      "fmin v17.8h, v17.8h, v21.8h\n"
      "fmin v18.8h, v18.8h, v21.8h\n"
      "fmin v19.8h, v19.8h, v21.8h\n"
      "fmax v8.8h, v8.8h, v20.8h\n"
      "fmax v9.8h, v9.8h, v20.8h\n"
      "fmax v10.8h, v10.8h, v20.8h\n"
      "fmax v11.8h, v11.8h, v20.8h\n"
      "fmax v12.8h, v12.8h, v20.8h\n"
      "fmax v13.8h, v13.8h, v20.8h\n"
      "fmax v14.8h, v14.8h, v20.8h\n"
      "fmax v15.8h, v15.8h, v20.8h\n"
      "fmax v16.8h, v16.8h, v20.8h\n"
      "fmax v17.8h, v17.8h, v20.8h\n"
      "fmax v18.8h, v18.8h, v20.8h\n"
      "fmax v19.8h, v19.8h, v20.8h\n"
      "132:"  // Height 3: No activation
      "cmp x14, #0x20\n"
      "bge 149f\n"
      "tbz x14, #4, 140f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v9.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v13.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v17.8h }, [x25], #0x10\n"
      "tbz x14, #3, 136f\n"
      "st1 { v10.8h }, [x13], #0x10\n"
      "st1 { v14.8h }, [x26], #0x10\n"
      "st1 { v18.8h }, [x25], #0x10\n"
      "tbz x14, #2, 134f\n"
      "str d11, [x13], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "tbz x14, #1, 133f\n"
      "st1 { v11.s }[2], [x13], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v19.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v11.h }[6], [x13]\n"
      "st1 { v15.h }[6], [x26]\n"
      "st1 { v19.h }[6], [x25]\n"
      "b 148f\n"
      "133:"  // Height 3: Partial direct writeback: partial_1_28
      "tbz x14, #0, 148f\n"
      "st1 { v11.h }[4], [x13]\n"
      "st1 { v15.h }[4], [x26]\n"
      "st1 { v19.h }[4], [x25]\n"
      "b 148f\n"
      "134:"  // Height 3: Partial direct writeback: partial_2_24
      "tbz x14, #1, 135f\n"
      "str s11, [x13], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s19, [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v11.h }[2], [x13]\n"
      "st1 { v15.h }[2], [x26]\n"
      "st1 { v19.h }[2], [x25]\n"
      "b 148f\n"
      "135:"  // Height 3: Partial direct writeback: partial_1_24
      "tbz x14, #0, 148f\n"
      "str h11, [x13, #0x0]\n"
      "str h15, [x26, #0x0]\n"
      "str h19, [x25, #0x0]\n"
      "b 148f\n"
      "136:"  // Height 3: Partial direct writeback: partial_4_16
      "tbz x14, #2, 138f\n"
      "str d10, [x13], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "tbz x14, #1, 137f\n"
      "st1 { v10.s }[2], [x13], #0x4\n"
      "st1 { v14.s }[2], [x26], #0x4\n"
      "st1 { v18.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v10.h }[6], [x13]\n"
      "st1 { v14.h }[6], [x26]\n"
      "st1 { v18.h }[6], [x25]\n"
      "b 148f\n"
      "137:"  // Height 3: Partial direct writeback: partial_1_20
      "tbz x14, #0, 148f\n"
      "st1 { v10.h }[4], [x13]\n"
      "st1 { v14.h }[4], [x26]\n"
      "st1 { v18.h }[4], [x25]\n"
      "b 148f\n"
      "138:"  // Height 3: Partial direct writeback: partial_2_16
      "tbz x14, #1, 139f\n"
      "str s10, [x13], #0x4\n"
      "str s14, [x26], #0x4\n"
      "str s18, [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v10.h }[2], [x13]\n"
      "st1 { v14.h }[2], [x26]\n"
      "st1 { v18.h }[2], [x25]\n"
      "b 148f\n"
      "139:"  // Height 3: Partial direct writeback: partial_1_16
      "tbz x14, #0, 148f\n"
      "str h10, [x13, #0x0]\n"
      "str h14, [x26, #0x0]\n"
      "str h18, [x25, #0x0]\n"
      "b 148f\n"
      "140:"  // Height 3: Partial direct writeback: partial_8_0
      "tbz x14, #3, 144f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "tbz x14, #2, 142f\n"
      "str d9, [x13], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "tbz x14, #1, 141f\n"
      "st1 { v9.s }[2], [x13], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v9.h }[6], [x13]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "b 148f\n"
      "141:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x14, #0, 148f\n"
      "st1 { v9.h }[4], [x13]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "b 148f\n"
      "142:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x14, #1, 143f\n"
      "str s9, [x13], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v9.h }[2], [x13]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "b 148f\n"
      "143:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x14, #0, 148f\n"
      "str h9, [x13, #0x0]\n"
      "str h13, [x26, #0x0]\n"
      "str h17, [x25, #0x0]\n"
      "b 148f\n"
      "144:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x14, #2, 146f\n"
      "str d8, [x13], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "tbz x14, #1, 145f\n"
      "st1 { v8.s }[2], [x13], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v8.h }[6], [x13]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "b 148f\n"
      "145:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x14, #0, 148f\n"
      "st1 { v8.h }[4], [x13]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "b 148f\n"
      "146:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x14, #1, 147f\n"
      "str s8, [x13], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "tbz x14, #0, 148f\n"
      "st1 { v8.h }[2], [x13]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "b 148f\n"
      "147:"  // Height 3: Partial direct writeback: partial_1_0
      "str h8, [x13, #0x0]\n"
      "str h12, [x26, #0x0]\n"
      "str h16, [x25, #0x0]\n"
      "148:"  // Height 3: Partial direct writeback: Done
      "b 150f\n"
      "149:"  // Height 3: Full writeback
      "str q8, [x13, #0x0]\n"
      "str q9, [x13, #0x10]\n"
      "str q10, [x13, #0x20]\n"
      "str q11, [x13, #0x30]\n"
      "add x13, x13, #0x40\n"
      "str q12, [x26, #0x0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0x0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "150:"  // Height 3: Writeback done
      "subs x14, x14, #0x20\n"
      "bgt 102b\n"
      "b 302f\n"
      "151:"  // Height 4
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "152:"  // Height 4: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x14, #0x18\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 153f\n"
      "cmp x14, #0x10\n"
      "mov x9, x12\n"
      "bgt 153f\n"
      "cmp x14, #0x8\n"
      "mov x10, x12\n"
      "bgt 153f\n"
      "mov x11, x12\n"
      "153:"  // Height 4: B setup done
      "cbz x15, 154f\n"
      "ldr q8, [x15, #0x0]\n"
      "ldr q9, [x15, #0x10]\n"
      "ldr q10, [x15, #0x20]\n"
      "ldr q11, [x15, #0x30]\n"
      "add x15, x15, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "b 173f\n"
      "154:"  // Height 4: no bias
      "tbz %x[flags], #0, 172f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x14, #0x20\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "bge 171f\n"
      "tbz x14, #4, 162f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v20.8h }, [x24], #0x10\n"
      "ld1 { v9.8h }, [x13], #0x10\n"
      "ld1 { v13.8h }, [x26], #0x10\n"
      "ld1 { v17.8h }, [x25], #0x10\n"
      "ld1 { v21.8h }, [x24], #0x10\n"
      "tbz x14, #3, 158f\n"
      "ld1 { v10.8h }, [x13], #0x10\n"
      "ld1 { v14.8h }, [x26], #0x10\n"
      "ld1 { v18.8h }, [x25], #0x10\n"
      "ld1 { v22.8h }, [x24], #0x10\n"
      "tbz x14, #2, 156f\n"
      "ldr d11, [x13], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "tbz x14, #1, 155f\n"
      "ld1 { v11.s }[2], [x13], #0x4\n"
      "ld1 { v15.s }[2], [x26], #0x4\n"
      "mov x20, #0x3c\n"
      "ld1 { v19.s }[2], [x25], #0x4\n"
      "ld1 { v23.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v11.h }[6], [x13]\n"
      "ld1 { v15.h }[6], [x26]\n"
      "ld1 { v19.h }[6], [x25]\n"
      "ld1 { v23.h }[6], [x24]\n"
      "b 170f\n"
      "155:"  // Height 4: Partial accumulate: partial_1_28
      "mov x20, #0x38\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v11.h }[4], [x13]\n"
      "ld1 { v15.h }[4], [x26]\n"
      "ld1 { v19.h }[4], [x25]\n"
      "ld1 { v23.h }[4], [x24]\n"
      "b 170f\n"
      "156:"  // Height 4: Partial accumulate: partial_2_24
      "tbz x14, #1, 157f\n"
      "ldr s11, [x13], #0x4\n"
      "ldr s15, [x26], #0x4\n"
      "mov x20, #0x34\n"
      "ldr s19, [x25], #0x4\n"
      "ldr s23, [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v11.h }[2], [x13]\n"
      "ld1 { v15.h }[2], [x26]\n"
      "ld1 { v19.h }[2], [x25]\n"
      "ld1 { v23.h }[2], [x24]\n"
      "b 170f\n"
      "157:"  // Height 4: Partial accumulate: partial_1_24
      "mov x20, #0x30\n"
      "tbz x14, #0, 170f\n"
      "ldr h11, [x13, #0x0]\n"
      "ldr h15, [x26, #0x0]\n"
      "ldr h19, [x25, #0x0]\n"
      "ldr h23, [x24, #0x0]\n"
      "b 170f\n"
      "158:"  // Height 4: Partial accumulate: partial_4_16
      "tbz x14, #2, 160f\n"
      "ldr d10, [x13], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "tbz x14, #1, 159f\n"
      "ld1 { v10.s }[2], [x13], #0x4\n"
      "ld1 { v14.s }[2], [x26], #0x4\n"
      "mov x20, #0x2c\n"
      "ld1 { v18.s }[2], [x25], #0x4\n"
      "ld1 { v22.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v10.h }[6], [x13]\n"
      "ld1 { v14.h }[6], [x26]\n"
      "ld1 { v18.h }[6], [x25]\n"
      "ld1 { v22.h }[6], [x24]\n"
      "b 170f\n"
      "159:"  // Height 4: Partial accumulate: partial_1_20
      "mov x20, #0x28\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v10.h }[4], [x13]\n"
      "ld1 { v14.h }[4], [x26]\n"
      "ld1 { v18.h }[4], [x25]\n"
      "ld1 { v22.h }[4], [x24]\n"
      "b 170f\n"
      "160:"  // Height 4: Partial accumulate: partial_2_16
      "tbz x14, #1, 161f\n"
      "ldr s10, [x13], #0x4\n"
      "ldr s14, [x26], #0x4\n"
      "mov x20, #0x24\n"
      "ldr s18, [x25], #0x4\n"
      "ldr s22, [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v10.h }[2], [x13]\n"
      "ld1 { v14.h }[2], [x26]\n"
      "ld1 { v18.h }[2], [x25]\n"
      "ld1 { v22.h }[2], [x24]\n"
      "b 170f\n"
      "161:"  // Height 4: Partial accumulate: partial_1_16
      "mov x20, #0x20\n"
      "tbz x14, #0, 170f\n"
      "ldr h10, [x13, #0x0]\n"
      "ldr h14, [x26, #0x0]\n"
      "ldr h18, [x25, #0x0]\n"
      "ldr h22, [x24, #0x0]\n"
      "b 170f\n"
      "162:"  // Height 4: Partial accumulate: partial_8_0
      "tbz x14, #3, 166f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v20.8h }, [x24], #0x10\n"
      "tbz x14, #2, 164f\n"
      "ldr d9, [x13], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "tbz x14, #1, 163f\n"
      "ld1 { v9.s }[2], [x13], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "ld1 { v21.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v9.h }[6], [x13]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "ld1 { v21.h }[6], [x24]\n"
      "b 170f\n"
      "163:"  // Height 4: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v9.h }[4], [x13]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "ld1 { v21.h }[4], [x24]\n"
      "b 170f\n"
      "164:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x14, #1, 165f\n"
      "ldr s9, [x13], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "ldr s21, [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v9.h }[2], [x13]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "ld1 { v21.h }[2], [x24]\n"
      "b 170f\n"
      "165:"  // Height 4: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x14, #0, 170f\n"
      "ldr h9, [x13, #0x0]\n"
      "ldr h13, [x26, #0x0]\n"
      "ldr h17, [x25, #0x0]\n"
      "ldr h21, [x24, #0x0]\n"
      "b 170f\n"
      "166:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x14, #2, 168f\n"
      "ldr d8, [x13], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "tbz x14, #1, 167f\n"
      "ld1 { v8.s }[2], [x13], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "ld1 { v20.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v8.h }[6], [x13]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "ld1 { v20.h }[6], [x24]\n"
      "b 170f\n"
      "167:"  // Height 4: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v8.h }[4], [x13]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "ld1 { v20.h }[4], [x24]\n"
      "b 170f\n"
      "168:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x14, #1, 169f\n"
      "ldr s8, [x13], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "tbz x14, #0, 170f\n"
      "ld1 { v8.h }[2], [x13]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "ld1 { v20.h }[2], [x24]\n"
      "b 170f\n"
      "169:"  // Height 4: Partial accumulate: partial_1_0
      "ldr h8, [x13, #0x0]\n"
      "ldr h12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr h16, [x25, #0x0]\n"
      "ldr h20, [x24, #0x0]\n"
      "170:"  // Height 4: Partial accumulate: Done
      "sub x13, x13, x20\n"
      "b 173f\n"
      "171:"  // Height 4: full accumulate
      "ldr q8, [x13, #0x0]\n"
      "ldr q9, [x13, #0x10]\n"
      "ldr q10, [x13, #0x20]\n"
      "ldr q11, [x13, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0x0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "ldr q20, [x24, #0x0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "b 173f\n"
      "172:"  // Height 4: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "173:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "174:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 175f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 176f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "b 176f\n"
      "175:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "176:"  // Height 4: input setup done
      "cmp x27, #0x8\n"
      "blt 179f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q6, [x12, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "blt 178f\n"
      "177:"  // Height 4: Multiply loop: Main loop head
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "fmla v20.8h, v6.8h, v3.h[0]\n"
      "ldr q25, [x10, #0x0]\n"
      "cmp x27, #0x10\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "fmla v21.8h, v7.8h, v3.h[0]\n"
      "ldr q24, [x9, #0x0]\n"
      "add x23, x23, #0x10\n"
      "fmla v10.8h, v25.8h, v0.h[0]\n"
      "fmla v14.8h, v25.8h, v1.h[0]\n"
      "fmla v18.8h, v25.8h, v2.h[0]\n"
      "fmla v22.8h, v25.8h, v3.h[0]\n"
      "ldr q25, [x12, #0x10]\n"
      "fmla v11.8h, v24.8h, v0.h[0]\n"
      "fmla v15.8h, v24.8h, v1.h[0]\n"
      "fmla v19.8h, v24.8h, v2.h[0]\n"
      "fmla v23.8h, v24.8h, v3.h[0]\n"
      "ldr q24, [x11, #0x10]\n"
      "fmla v8.8h, v25.8h, v0.h[1]\n"
      "fmla v12.8h, v25.8h, v1.h[1]\n"
      "fmla v16.8h, v25.8h, v2.h[1]\n"
      "fmla v20.8h, v25.8h, v3.h[1]\n"
      "ldr q25, [x10, #0x10]\n"
      "fmla v9.8h, v24.8h, v0.h[1]\n"
      "fmla v13.8h, v24.8h, v1.h[1]\n"
      "fmla v17.8h, v24.8h, v2.h[1]\n"
      "fmla v21.8h, v24.8h, v3.h[1]\n"
      "ldr q24, [x9, #0x10]\n"
      "fmla v10.8h, v25.8h, v0.h[1]\n"
      "fmla v14.8h, v25.8h, v1.h[1]\n"
      "fmla v18.8h, v25.8h, v2.h[1]\n"
      "fmla v22.8h, v25.8h, v3.h[1]\n"
      "ldr q25, [x12, #0x20]\n"
      "fmla v11.8h, v24.8h, v0.h[1]\n"
      "fmla v15.8h, v24.8h, v1.h[1]\n"
      "fmla v19.8h, v24.8h, v2.h[1]\n"
      "fmla v23.8h, v24.8h, v3.h[1]\n"
      "ldr q24, [x11, #0x20]\n"
      "fmla v8.8h, v25.8h, v0.h[2]\n"
      "fmla v12.8h, v25.8h, v1.h[2]\n"
      "fmla v16.8h, v25.8h, v2.h[2]\n"
      "fmla v20.8h, v25.8h, v3.h[2]\n"
      "ldr q25, [x10, #0x20]\n"
      "fmla v9.8h, v24.8h, v0.h[2]\n"
      "fmla v13.8h, v24.8h, v1.h[2]\n"
      "fmla v17.8h, v24.8h, v2.h[2]\n"
      "fmla v21.8h, v24.8h, v3.h[2]\n"
      "ldr q24, [x9, #0x20]\n"
      "fmla v10.8h, v25.8h, v0.h[2]\n"
      "fmla v14.8h, v25.8h, v1.h[2]\n"
      "fmla v18.8h, v25.8h, v2.h[2]\n"
      "fmla v22.8h, v25.8h, v3.h[2]\n"
      "ldr q25, [x12, #0x30]\n"
      "fmla v11.8h, v24.8h, v0.h[2]\n"
      "fmla v15.8h, v24.8h, v1.h[2]\n"
      "fmla v19.8h, v24.8h, v2.h[2]\n"
      "fmla v23.8h, v24.8h, v3.h[2]\n"
      "ldr q24, [x11, #0x30]\n"
      "fmla v8.8h, v25.8h, v0.h[3]\n"
      "fmla v12.8h, v25.8h, v1.h[3]\n"
      "fmla v16.8h, v25.8h, v2.h[3]\n"
      "fmla v20.8h, v25.8h, v3.h[3]\n"
      "ldr q25, [x10, #0x30]\n"
      "fmla v9.8h, v24.8h, v0.h[3]\n"
      "fmla v13.8h, v24.8h, v1.h[3]\n"
      "fmla v17.8h, v24.8h, v2.h[3]\n"
      "fmla v21.8h, v24.8h, v3.h[3]\n"
      "ldr q24, [x9, #0x30]\n"
      "fmla v10.8h, v25.8h, v0.h[3]\n"
      "fmla v14.8h, v25.8h, v1.h[3]\n"
      "fmla v18.8h, v25.8h, v2.h[3]\n"
      "fmla v22.8h, v25.8h, v3.h[3]\n"
      "ldr q25, [x12, #0x40]\n"
      "fmla v11.8h, v24.8h, v0.h[3]\n"
      "fmla v15.8h, v24.8h, v1.h[3]\n"
      "fmla v19.8h, v24.8h, v2.h[3]\n"
      "fmla v23.8h, v24.8h, v3.h[3]\n"
      "ldr q24, [x11, #0x40]\n"
      "fmla v8.8h, v25.8h, v0.h[4]\n"
      "fmla v12.8h, v25.8h, v1.h[4]\n"
      "fmla v16.8h, v25.8h, v2.h[4]\n"
      "fmla v20.8h, v25.8h, v3.h[4]\n"
      "ldr q25, [x10, #0x40]\n"
      "fmla v9.8h, v24.8h, v0.h[4]\n"
      "fmla v13.8h, v24.8h, v1.h[4]\n"
      "fmla v17.8h, v24.8h, v2.h[4]\n"
      "fmla v21.8h, v24.8h, v3.h[4]\n"
      "ldr q24, [x9, #0x40]\n"
      "fmla v10.8h, v25.8h, v0.h[4]\n"
      "fmla v14.8h, v25.8h, v1.h[4]\n"
      "fmla v18.8h, v25.8h, v2.h[4]\n"
      "fmla v22.8h, v25.8h, v3.h[4]\n"
      "ldr q25, [x12, #0x50]\n"
      "fmla v11.8h, v24.8h, v0.h[4]\n"
      "fmla v15.8h, v24.8h, v1.h[4]\n"
      "fmla v19.8h, v24.8h, v2.h[4]\n"
      "fmla v23.8h, v24.8h, v3.h[4]\n"
      "ldr q24, [x11, #0x50]\n"
      "fmla v8.8h, v25.8h, v0.h[5]\n"
      "fmla v12.8h, v25.8h, v1.h[5]\n"
      "fmla v16.8h, v25.8h, v2.h[5]\n"
      "fmla v20.8h, v25.8h, v3.h[5]\n"
      "ldr q25, [x10, #0x50]\n"
      "fmla v9.8h, v24.8h, v0.h[5]\n"
      "fmla v13.8h, v24.8h, v1.h[5]\n"
      "fmla v17.8h, v24.8h, v2.h[5]\n"
      "fmla v21.8h, v24.8h, v3.h[5]\n"
      "ldr q24, [x9, #0x50]\n"
      "fmla v10.8h, v25.8h, v0.h[5]\n"
      "fmla v14.8h, v25.8h, v1.h[5]\n"
      "fmla v18.8h, v25.8h, v2.h[5]\n"
      "fmla v22.8h, v25.8h, v3.h[5]\n"
      "ldr q25, [x12, #0x60]\n"
      "fmla v11.8h, v24.8h, v0.h[5]\n"
      "fmla v15.8h, v24.8h, v1.h[5]\n"
      "fmla v19.8h, v24.8h, v2.h[5]\n"
      "fmla v23.8h, v24.8h, v3.h[5]\n"
      "ldr q24, [x11, #0x60]\n"
      "fmla v8.8h, v25.8h, v0.h[6]\n"
      "fmla v12.8h, v25.8h, v1.h[6]\n"
      "fmla v16.8h, v25.8h, v2.h[6]\n"
      "fmla v20.8h, v25.8h, v3.h[6]\n"
      "ldr q25, [x10, #0x60]\n"
      "fmla v9.8h, v24.8h, v0.h[6]\n"
      "fmla v13.8h, v24.8h, v1.h[6]\n"
      "fmla v17.8h, v24.8h, v2.h[6]\n"
      "fmla v21.8h, v24.8h, v3.h[6]\n"
      "ldr q24, [x9, #0x60]\n"
      "fmla v10.8h, v25.8h, v0.h[6]\n"
      "fmla v14.8h, v25.8h, v1.h[6]\n"
      "fmla v18.8h, v25.8h, v2.h[6]\n"
      "fmla v22.8h, v25.8h, v3.h[6]\n"
      "ldr q25, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v24.8h, v0.h[6]\n"
      "fmla v15.8h, v24.8h, v1.h[6]\n"
      "fmla v19.8h, v24.8h, v2.h[6]\n"
      "fmla v23.8h, v24.8h, v3.h[6]\n"
      "ldr q24, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v25.8h, v0.h[7]\n"
      "fmla v12.8h, v25.8h, v1.h[7]\n"
      "fmla v16.8h, v25.8h, v2.h[7]\n"
      "fmla v20.8h, v25.8h, v3.h[7]\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v24.8h, v0.h[7]\n"
      "fmla v13.8h, v24.8h, v1.h[7]\n"
      "fmla v17.8h, v24.8h, v2.h[7]\n"
      "fmla v21.8h, v24.8h, v3.h[7]\n"
      "ldr q24, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v25.8h, v0.h[7]\n"
      "fmla v14.8h, v25.8h, v1.h[7]\n"
      "fmla v18.8h, v25.8h, v2.h[7]\n"
      "fmla v22.8h, v25.8h, v3.h[7]\n"
      "ldr q6, [x12, #0x0]\n"
      "fmla v11.8h, v24.8h, v0.h[7]\n"
      "ldr q0, [x26, #0x0]\n"
      "fmla v15.8h, v24.8h, v1.h[7]\n"
      "ldr q1, [x25, #0x0]\n"
      "fmla v19.8h, v24.8h, v2.h[7]\n"
      "ldr q2, [x24, #0x0]\n"
      "fmla v23.8h, v24.8h, v3.h[7]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "bge 177b\n"
      "178:"  // Height 4: Multiply loop: Single iteration only
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "fmla v20.8h, v6.8h, v3.h[0]\n"
      "ldr q25, [x10, #0x0]\n"
      "add x25, x25, #0x10\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "fmla v21.8h, v7.8h, v3.h[0]\n"
      "ldr q24, [x9, #0x0]\n"
      "fmla v10.8h, v25.8h, v0.h[0]\n"
      "fmla v14.8h, v25.8h, v1.h[0]\n"
      "fmla v18.8h, v25.8h, v2.h[0]\n"
      "fmla v22.8h, v25.8h, v3.h[0]\n"
      "ldr q25, [x12, #0x10]\n"
      "fmla v11.8h, v24.8h, v0.h[0]\n"
      "fmla v15.8h, v24.8h, v1.h[0]\n"
      "fmla v19.8h, v24.8h, v2.h[0]\n"
      "fmla v23.8h, v24.8h, v3.h[0]\n"
      "ldr q24, [x11, #0x10]\n"
      "fmla v8.8h, v25.8h, v0.h[1]\n"
      "fmla v12.8h, v25.8h, v1.h[1]\n"
      "fmla v16.8h, v25.8h, v2.h[1]\n"
      "fmla v20.8h, v25.8h, v3.h[1]\n"
      "ldr q25, [x10, #0x10]\n"
      "fmla v9.8h, v24.8h, v0.h[1]\n"
      "fmla v13.8h, v24.8h, v1.h[1]\n"
      "fmla v17.8h, v24.8h, v2.h[1]\n"
      "fmla v21.8h, v24.8h, v3.h[1]\n"
      "ldr q24, [x9, #0x10]\n"
      "fmla v10.8h, v25.8h, v0.h[1]\n"
      "fmla v14.8h, v25.8h, v1.h[1]\n"
      "fmla v18.8h, v25.8h, v2.h[1]\n"
      "fmla v22.8h, v25.8h, v3.h[1]\n"
      "ldr q25, [x12, #0x20]\n"
      "fmla v11.8h, v24.8h, v0.h[1]\n"
      "fmla v15.8h, v24.8h, v1.h[1]\n"
      "fmla v19.8h, v24.8h, v2.h[1]\n"
      "fmla v23.8h, v24.8h, v3.h[1]\n"
      "ldr q24, [x11, #0x20]\n"
      "fmla v8.8h, v25.8h, v0.h[2]\n"
      "fmla v12.8h, v25.8h, v1.h[2]\n"
      "fmla v16.8h, v25.8h, v2.h[2]\n"
      "fmla v20.8h, v25.8h, v3.h[2]\n"
      "ldr q25, [x10, #0x20]\n"
      "fmla v9.8h, v24.8h, v0.h[2]\n"
      "fmla v13.8h, v24.8h, v1.h[2]\n"
      "fmla v17.8h, v24.8h, v2.h[2]\n"
      "fmla v21.8h, v24.8h, v3.h[2]\n"
      "ldr q24, [x9, #0x20]\n"
      "fmla v10.8h, v25.8h, v0.h[2]\n"
      "fmla v14.8h, v25.8h, v1.h[2]\n"
      "fmla v18.8h, v25.8h, v2.h[2]\n"
      "fmla v22.8h, v25.8h, v3.h[2]\n"
      "ldr q25, [x12, #0x30]\n"
      "fmla v11.8h, v24.8h, v0.h[2]\n"
      "fmla v15.8h, v24.8h, v1.h[2]\n"
      "fmla v19.8h, v24.8h, v2.h[2]\n"
      "fmla v23.8h, v24.8h, v3.h[2]\n"
      "ldr q24, [x11, #0x30]\n"
      "fmla v8.8h, v25.8h, v0.h[3]\n"
      "fmla v12.8h, v25.8h, v1.h[3]\n"
      "fmla v16.8h, v25.8h, v2.h[3]\n"
      "fmla v20.8h, v25.8h, v3.h[3]\n"
      "ldr q25, [x10, #0x30]\n"
      "fmla v9.8h, v24.8h, v0.h[3]\n"
      "fmla v13.8h, v24.8h, v1.h[3]\n"
      "fmla v17.8h, v24.8h, v2.h[3]\n"
      "fmla v21.8h, v24.8h, v3.h[3]\n"
      "ldr q24, [x9, #0x30]\n"
      "fmla v10.8h, v25.8h, v0.h[3]\n"
      "fmla v14.8h, v25.8h, v1.h[3]\n"
      "fmla v18.8h, v25.8h, v2.h[3]\n"
      "fmla v22.8h, v25.8h, v3.h[3]\n"
      "ldr q25, [x12, #0x40]\n"
      "fmla v11.8h, v24.8h, v0.h[3]\n"
      "fmla v15.8h, v24.8h, v1.h[3]\n"
      "fmla v19.8h, v24.8h, v2.h[3]\n"
      "fmla v23.8h, v24.8h, v3.h[3]\n"
      "ldr q24, [x11, #0x40]\n"
      "fmla v8.8h, v25.8h, v0.h[4]\n"
      "fmla v12.8h, v25.8h, v1.h[4]\n"
      "fmla v16.8h, v25.8h, v2.h[4]\n"
      "fmla v20.8h, v25.8h, v3.h[4]\n"
      "ldr q25, [x10, #0x40]\n"
      "fmla v9.8h, v24.8h, v0.h[4]\n"
      "fmla v13.8h, v24.8h, v1.h[4]\n"
      "fmla v17.8h, v24.8h, v2.h[4]\n"
      "fmla v21.8h, v24.8h, v3.h[4]\n"
      "ldr q24, [x9, #0x40]\n"
      "fmla v10.8h, v25.8h, v0.h[4]\n"
      "fmla v14.8h, v25.8h, v1.h[4]\n"
      "fmla v18.8h, v25.8h, v2.h[4]\n"
      "fmla v22.8h, v25.8h, v3.h[4]\n"
      "ldr q25, [x12, #0x50]\n"
      "fmla v11.8h, v24.8h, v0.h[4]\n"
      "fmla v15.8h, v24.8h, v1.h[4]\n"
      "fmla v19.8h, v24.8h, v2.h[4]\n"
      "fmla v23.8h, v24.8h, v3.h[4]\n"
      "ldr q24, [x11, #0x50]\n"
      "fmla v8.8h, v25.8h, v0.h[5]\n"
      "fmla v12.8h, v25.8h, v1.h[5]\n"
      "fmla v16.8h, v25.8h, v2.h[5]\n"
      "fmla v20.8h, v25.8h, v3.h[5]\n"
      "ldr q25, [x10, #0x50]\n"
      "fmla v9.8h, v24.8h, v0.h[5]\n"
      "fmla v13.8h, v24.8h, v1.h[5]\n"
      "fmla v17.8h, v24.8h, v2.h[5]\n"
      "fmla v21.8h, v24.8h, v3.h[5]\n"
      "ldr q24, [x9, #0x50]\n"
      "fmla v10.8h, v25.8h, v0.h[5]\n"
      "fmla v14.8h, v25.8h, v1.h[5]\n"
      "fmla v18.8h, v25.8h, v2.h[5]\n"
      "fmla v22.8h, v25.8h, v3.h[5]\n"
      "ldr q25, [x12, #0x60]\n"
      "fmla v11.8h, v24.8h, v0.h[5]\n"
      "fmla v15.8h, v24.8h, v1.h[5]\n"
      "fmla v19.8h, v24.8h, v2.h[5]\n"
      "fmla v23.8h, v24.8h, v3.h[5]\n"
      "ldr q24, [x11, #0x60]\n"
      "fmla v8.8h, v25.8h, v0.h[6]\n"
      "fmla v12.8h, v25.8h, v1.h[6]\n"
      "fmla v16.8h, v25.8h, v2.h[6]\n"
      "fmla v20.8h, v25.8h, v3.h[6]\n"
      "ldr q25, [x10, #0x60]\n"
      "fmla v9.8h, v24.8h, v0.h[6]\n"
      "fmla v13.8h, v24.8h, v1.h[6]\n"
      "fmla v17.8h, v24.8h, v2.h[6]\n"
      "fmla v21.8h, v24.8h, v3.h[6]\n"
      "ldr q24, [x9, #0x60]\n"
      "fmla v10.8h, v25.8h, v0.h[6]\n"
      "fmla v14.8h, v25.8h, v1.h[6]\n"
      "fmla v18.8h, v25.8h, v2.h[6]\n"
      "fmla v22.8h, v25.8h, v3.h[6]\n"
      "ldr q25, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v24.8h, v0.h[6]\n"
      "fmla v15.8h, v24.8h, v1.h[6]\n"
      "fmla v19.8h, v24.8h, v2.h[6]\n"
      "fmla v23.8h, v24.8h, v3.h[6]\n"
      "ldr q24, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v25.8h, v0.h[7]\n"
      "fmla v12.8h, v25.8h, v1.h[7]\n"
      "fmla v16.8h, v25.8h, v2.h[7]\n"
      "fmla v20.8h, v25.8h, v3.h[7]\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v24.8h, v0.h[7]\n"
      "fmla v13.8h, v24.8h, v1.h[7]\n"
      "fmla v17.8h, v24.8h, v2.h[7]\n"
      "fmla v21.8h, v24.8h, v3.h[7]\n"
      "ldr q24, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v25.8h, v0.h[7]\n"
      "fmla v14.8h, v25.8h, v1.h[7]\n"
      "fmla v18.8h, v25.8h, v2.h[7]\n"
      "fmla v22.8h, v25.8h, v3.h[7]\n"
      "fmla v11.8h, v24.8h, v0.h[7]\n"
      "fmla v15.8h, v24.8h, v1.h[7]\n"
      "fmla v19.8h, v24.8h, v2.h[7]\n"
      "fmla v23.8h, v24.8h, v3.h[7]\n"
      "179:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 181f\n"
      "180:"  // Height 4: Multiply loop: Odd block loop
      "ldr h3, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h0, [x23], #0x2\n"
      "ldr q25, [x12, #0x0]\n"
      "ldr q24, [x11, #0x0]\n"
      "add x12, x12, #0x10\n"
      "add x11, x11, #0x10\n"
      "fmla v8.8h, v25.8h, v3.h[0]\n"
      "fmla v12.8h, v25.8h, v2.h[0]\n"
      "fmla v16.8h, v25.8h, v1.h[0]\n"
      "fmla v20.8h, v25.8h, v0.h[0]\n"
      "ldr q25, [x10, #0x0]\n"
      "add x10, x10, #0x10\n"
      "fmla v9.8h, v24.8h, v3.h[0]\n"
      "fmla v13.8h, v24.8h, v2.h[0]\n"
      "fmla v17.8h, v24.8h, v1.h[0]\n"
      "fmla v21.8h, v24.8h, v0.h[0]\n"
      "ldr q24, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "fmla v10.8h, v25.8h, v3.h[0]\n"
      "fmla v14.8h, v25.8h, v2.h[0]\n"
      "fmla v18.8h, v25.8h, v1.h[0]\n"
      "fmla v22.8h, v25.8h, v0.h[0]\n"
      "fmla v11.8h, v24.8h, v3.h[0]\n"
      "fmla v15.8h, v24.8h, v2.h[0]\n"
      "fmla v19.8h, v24.8h, v1.h[0]\n"
      "fmla v23.8h, v24.8h, v0.h[0]\n"
      "cbnz x27, 180b\n"
      "181:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 174b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "tbz %x[flags], #1, 182f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v25.8h }, [x21]\n"
      "ld1r { v24.8h }, [x20]\n"
      "fmin v8.8h, v8.8h, v25.8h\n"
      "fmin v9.8h, v9.8h, v25.8h\n"
      "fmin v10.8h, v10.8h, v25.8h\n"
      "fmin v11.8h, v11.8h, v25.8h\n"
      "fmin v12.8h, v12.8h, v25.8h\n"
      "fmin v13.8h, v13.8h, v25.8h\n"
      "fmin v14.8h, v14.8h, v25.8h\n"
      "fmin v15.8h, v15.8h, v25.8h\n"
      "fmin v16.8h, v16.8h, v25.8h\n"
      "fmin v17.8h, v17.8h, v25.8h\n"
      "fmin v18.8h, v18.8h, v25.8h\n"
      "fmin v19.8h, v19.8h, v25.8h\n"
      "fmin v20.8h, v20.8h, v25.8h\n"
      "fmin v21.8h, v21.8h, v25.8h\n"
      "fmin v22.8h, v22.8h, v25.8h\n"
      "fmin v23.8h, v23.8h, v25.8h\n"
      "fmax v8.8h, v8.8h, v24.8h\n"
      "fmax v9.8h, v9.8h, v24.8h\n"
      "fmax v10.8h, v10.8h, v24.8h\n"
      "fmax v11.8h, v11.8h, v24.8h\n"
      "fmax v12.8h, v12.8h, v24.8h\n"
      "fmax v13.8h, v13.8h, v24.8h\n"
      "fmax v14.8h, v14.8h, v24.8h\n"
      "fmax v15.8h, v15.8h, v24.8h\n"
      "fmax v16.8h, v16.8h, v24.8h\n"
      "fmax v17.8h, v17.8h, v24.8h\n"
      "fmax v18.8h, v18.8h, v24.8h\n"
      "fmax v19.8h, v19.8h, v24.8h\n"
      "fmax v20.8h, v20.8h, v24.8h\n"
      "fmax v21.8h, v21.8h, v24.8h\n"
      "fmax v22.8h, v22.8h, v24.8h\n"
      "fmax v23.8h, v23.8h, v24.8h\n"
      "182:"  // Height 4: No activation
      "cmp x14, #0x20\n"
      "bge 199f\n"
      "tbz x14, #4, 190f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v9.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v13.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v17.8h }, [x25], #0x10\n"
      "st1 { v20.8h }, [x24], #0x10\n"
      "st1 { v21.8h }, [x24], #0x10\n"
      "tbz x14, #3, 186f\n"
      "st1 { v10.8h }, [x13], #0x10\n"
      "st1 { v14.8h }, [x26], #0x10\n"
      "st1 { v18.8h }, [x25], #0x10\n"
      "st1 { v22.8h }, [x24], #0x10\n"
      "tbz x14, #2, 184f\n"
      "str d11, [x13], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "tbz x14, #1, 183f\n"
      "st1 { v11.s }[2], [x13], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v19.s }[2], [x25], #0x4\n"
      "st1 { v23.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v11.h }[6], [x13]\n"
      "st1 { v15.h }[6], [x26]\n"
      "st1 { v19.h }[6], [x25]\n"
      "st1 { v23.h }[6], [x24]\n"
      "b 198f\n"
      "183:"  // Height 4: Partial direct writeback: partial_1_28
      "tbz x14, #0, 198f\n"
      "st1 { v11.h }[4], [x13]\n"
      "st1 { v15.h }[4], [x26]\n"
      "st1 { v19.h }[4], [x25]\n"
      "st1 { v23.h }[4], [x24]\n"
      "b 198f\n"
      "184:"  // Height 4: Partial direct writeback: partial_2_24
      "tbz x14, #1, 185f\n"
      "str s11, [x13], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s19, [x25], #0x4\n"
      "str s23, [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v11.h }[2], [x13]\n"
      "st1 { v15.h }[2], [x26]\n"
      "st1 { v19.h }[2], [x25]\n"
      "st1 { v23.h }[2], [x24]\n"
      "b 198f\n"
      "185:"  // Height 4: Partial direct writeback: partial_1_24
      "tbz x14, #0, 198f\n"
      "str h11, [x13, #0x0]\n"
      "str h15, [x26, #0x0]\n"
      "str h19, [x25, #0x0]\n"
      "str h23, [x24, #0x0]\n"
      "b 198f\n"
      "186:"  // Height 4: Partial direct writeback: partial_4_16
      "tbz x14, #2, 188f\n"
      "str d10, [x13], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "str d22, [x24], #0x8\n"
      "tbz x14, #1, 187f\n"
      "st1 { v10.s }[2], [x13], #0x4\n"
      "st1 { v14.s }[2], [x26], #0x4\n"
      "st1 { v18.s }[2], [x25], #0x4\n"
      "st1 { v22.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v10.h }[6], [x13]\n"
      "st1 { v14.h }[6], [x26]\n"
      "st1 { v18.h }[6], [x25]\n"
      "st1 { v22.h }[6], [x24]\n"
      "b 198f\n"
      "187:"  // Height 4: Partial direct writeback: partial_1_20
      "tbz x14, #0, 198f\n"
      "st1 { v10.h }[4], [x13]\n"
      "st1 { v14.h }[4], [x26]\n"
      "st1 { v18.h }[4], [x25]\n"
      "st1 { v22.h }[4], [x24]\n"
      "b 198f\n"
      "188:"  // Height 4: Partial direct writeback: partial_2_16
      "tbz x14, #1, 189f\n"
      "str s10, [x13], #0x4\n"
      "str s14, [x26], #0x4\n"
      "str s18, [x25], #0x4\n"
      "str s22, [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v10.h }[2], [x13]\n"
      "st1 { v14.h }[2], [x26]\n"
      "st1 { v18.h }[2], [x25]\n"
      "st1 { v22.h }[2], [x24]\n"
      "b 198f\n"
      "189:"  // Height 4: Partial direct writeback: partial_1_16
      "tbz x14, #0, 198f\n"
      "str h10, [x13, #0x0]\n"
      "str h14, [x26, #0x0]\n"
      "str h18, [x25, #0x0]\n"
      "str h22, [x24, #0x0]\n"
      "b 198f\n"
      "190:"  // Height 4: Partial direct writeback: partial_8_0
      "tbz x14, #3, 194f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v20.8h }, [x24], #0x10\n"
      "tbz x14, #2, 192f\n"
      "str d9, [x13], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "tbz x14, #1, 191f\n"
      "st1 { v9.s }[2], [x13], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "st1 { v21.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v9.h }[6], [x13]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "st1 { v21.h }[6], [x24]\n"
      "b 198f\n"
      "191:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x14, #0, 198f\n"
      "st1 { v9.h }[4], [x13]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "st1 { v21.h }[4], [x24]\n"
      "b 198f\n"
      "192:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x14, #1, 193f\n"
      "str s9, [x13], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "str s21, [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v9.h }[2], [x13]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "st1 { v21.h }[2], [x24]\n"
      "b 198f\n"
      "193:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x14, #0, 198f\n"
      "str h9, [x13, #0x0]\n"
      "str h13, [x26, #0x0]\n"
      "str h17, [x25, #0x0]\n"
      "str h21, [x24, #0x0]\n"
      "b 198f\n"
      "194:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x14, #2, 196f\n"
      "str d8, [x13], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "tbz x14, #1, 195f\n"
      "st1 { v8.s }[2], [x13], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v8.h }[6], [x13]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "st1 { v20.h }[6], [x24]\n"
      "b 198f\n"
      "195:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x14, #0, 198f\n"
      "st1 { v8.h }[4], [x13]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "st1 { v20.h }[4], [x24]\n"
      "b 198f\n"
      "196:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x14, #1, 197f\n"
      "str s8, [x13], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s20, [x24], #0x4\n"
      "tbz x14, #0, 198f\n"
      "st1 { v8.h }[2], [x13]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "st1 { v20.h }[2], [x24]\n"
      "b 198f\n"
      "197:"  // Height 4: Partial direct writeback: partial_1_0
      "str h8, [x13, #0x0]\n"
      "str h12, [x26, #0x0]\n"
      "str h16, [x25, #0x0]\n"
      "str h20, [x24, #0x0]\n"
      "198:"  // Height 4: Partial direct writeback: Done
      "b 200f\n"
      "199:"  // Height 4: Full writeback
      "str q8, [x13, #0x0]\n"
      "str q9, [x13, #0x10]\n"
      "str q10, [x13, #0x20]\n"
      "str q11, [x13, #0x30]\n"
      "add x13, x13, #0x40\n"
      "str q12, [x26, #0x0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0x0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "str q20, [x24, #0x0]\n"
      "str q21, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "str q23, [x24, #0x30]\n"
      "200:"  // Height 4: Writeback done
      "subs x14, x14, #0x20\n"
      "bgt 152b\n"
      "b 302f\n"
      "201:"  // Height 5
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "202:"  // Height 5: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x14, #0x18\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 203f\n"
      "cmp x14, #0x10\n"
      "mov x9, x12\n"
      "bgt 203f\n"
      "cmp x14, #0x8\n"
      "mov x10, x12\n"
      "bgt 203f\n"
      "mov x11, x12\n"
      "203:"  // Height 5: B setup done
      "cbz x15, 204f\n"
      "ldr q8, [x15, #0x0]\n"
      "ldr q9, [x15, #0x10]\n"
      "ldr q10, [x15, #0x20]\n"
      "ldr q11, [x15, #0x30]\n"
      "add x15, x15, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "b 223f\n"
      "204:"  // Height 5: no bias
      "tbz %x[flags], #0, 222f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x14, #0x20\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "bge 221f\n"
      "tbz x14, #4, 212f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v20.8h }, [x24], #0x10\n"
      "ld1 { v24.8h }, [x23], #0x10\n"
      "ld1 { v9.8h }, [x13], #0x10\n"
      "ld1 { v13.8h }, [x26], #0x10\n"
      "ld1 { v17.8h }, [x25], #0x10\n"
      "ld1 { v21.8h }, [x24], #0x10\n"
      "ld1 { v25.8h }, [x23], #0x10\n"
      "tbz x14, #3, 208f\n"
      "ld1 { v10.8h }, [x13], #0x10\n"
      "ld1 { v14.8h }, [x26], #0x10\n"
      "ld1 { v18.8h }, [x25], #0x10\n"
      "ld1 { v22.8h }, [x24], #0x10\n"
      "ld1 { v26.8h }, [x23], #0x10\n"
      "tbz x14, #2, 206f\n"
      "ldr d11, [x13], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "ldr d27, [x23], #0x8\n"
      "tbz x14, #1, 205f\n"
      "ld1 { v11.s }[2], [x13], #0x4\n"
      "ld1 { v15.s }[2], [x26], #0x4\n"
      "mov x20, #0x3c\n"
      "ld1 { v19.s }[2], [x25], #0x4\n"
      "ld1 { v23.s }[2], [x24], #0x4\n"
      "ld1 { v27.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v11.h }[6], [x13]\n"
      "ld1 { v15.h }[6], [x26]\n"
      "ld1 { v19.h }[6], [x25]\n"
      "ld1 { v23.h }[6], [x24]\n"
      "ld1 { v27.h }[6], [x23]\n"
      "b 220f\n"
      "205:"  // Height 5: Partial accumulate: partial_1_28
      "mov x20, #0x38\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v11.h }[4], [x13]\n"
      "ld1 { v15.h }[4], [x26]\n"
      "ld1 { v19.h }[4], [x25]\n"
      "ld1 { v23.h }[4], [x24]\n"
      "ld1 { v27.h }[4], [x23]\n"
      "b 220f\n"
      "206:"  // Height 5: Partial accumulate: partial_2_24
      "tbz x14, #1, 207f\n"
      "ldr s11, [x13], #0x4\n"
      "ldr s15, [x26], #0x4\n"
      "mov x20, #0x34\n"
      "ldr s19, [x25], #0x4\n"
      "ldr s23, [x24], #0x4\n"
      "ldr s27, [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v11.h }[2], [x13]\n"
      "ld1 { v15.h }[2], [x26]\n"
      "ld1 { v19.h }[2], [x25]\n"
      "ld1 { v23.h }[2], [x24]\n"
      "ld1 { v27.h }[2], [x23]\n"
      "b 220f\n"
      "207:"  // Height 5: Partial accumulate: partial_1_24
      "mov x20, #0x30\n"
      "tbz x14, #0, 220f\n"
      "ldr h11, [x13, #0x0]\n"
      "ldr h15, [x26, #0x0]\n"
      "ldr h19, [x25, #0x0]\n"
      "ldr h23, [x24, #0x0]\n"
      "ldr h27, [x23, #0x0]\n"
      "b 220f\n"
      "208:"  // Height 5: Partial accumulate: partial_4_16
      "tbz x14, #2, 210f\n"
      "ldr d10, [x13], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "ldr d26, [x23], #0x8\n"
      "tbz x14, #1, 209f\n"
      "ld1 { v10.s }[2], [x13], #0x4\n"
      "ld1 { v14.s }[2], [x26], #0x4\n"
      "mov x20, #0x2c\n"
      "ld1 { v18.s }[2], [x25], #0x4\n"
      "ld1 { v22.s }[2], [x24], #0x4\n"
      "ld1 { v26.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v10.h }[6], [x13]\n"
      "ld1 { v14.h }[6], [x26]\n"
      "ld1 { v18.h }[6], [x25]\n"
      "ld1 { v22.h }[6], [x24]\n"
      "ld1 { v26.h }[6], [x23]\n"
      "b 220f\n"
      "209:"  // Height 5: Partial accumulate: partial_1_20
      "mov x20, #0x28\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v10.h }[4], [x13]\n"
      "ld1 { v14.h }[4], [x26]\n"
      "ld1 { v18.h }[4], [x25]\n"
      "ld1 { v22.h }[4], [x24]\n"
      "ld1 { v26.h }[4], [x23]\n"
      "b 220f\n"
      "210:"  // Height 5: Partial accumulate: partial_2_16
      "tbz x14, #1, 211f\n"
      "ldr s10, [x13], #0x4\n"
      "ldr s14, [x26], #0x4\n"
      "mov x20, #0x24\n"
      "ldr s18, [x25], #0x4\n"
      "ldr s22, [x24], #0x4\n"
      "ldr s26, [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v10.h }[2], [x13]\n"
      "ld1 { v14.h }[2], [x26]\n"
      "ld1 { v18.h }[2], [x25]\n"
      "ld1 { v22.h }[2], [x24]\n"
      "ld1 { v26.h }[2], [x23]\n"
      "b 220f\n"
      "211:"  // Height 5: Partial accumulate: partial_1_16
      "mov x20, #0x20\n"
      "tbz x14, #0, 220f\n"
      "ldr h10, [x13, #0x0]\n"
      "ldr h14, [x26, #0x0]\n"
      "ldr h18, [x25, #0x0]\n"
      "ldr h22, [x24, #0x0]\n"
      "ldr h26, [x23, #0x0]\n"
      "b 220f\n"
      "212:"  // Height 5: Partial accumulate: partial_8_0
      "tbz x14, #3, 216f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v20.8h }, [x24], #0x10\n"
      "ld1 { v24.8h }, [x23], #0x10\n"
      "tbz x14, #2, 214f\n"
      "ldr d9, [x13], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "tbz x14, #1, 213f\n"
      "ld1 { v9.s }[2], [x13], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "ld1 { v21.s }[2], [x24], #0x4\n"
      "ld1 { v25.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v9.h }[6], [x13]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "ld1 { v21.h }[6], [x24]\n"
      "ld1 { v25.h }[6], [x23]\n"
      "b 220f\n"
      "213:"  // Height 5: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v9.h }[4], [x13]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "ld1 { v21.h }[4], [x24]\n"
      "ld1 { v25.h }[4], [x23]\n"
      "b 220f\n"
      "214:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x14, #1, 215f\n"
      "ldr s9, [x13], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "ldr s21, [x24], #0x4\n"
      "ldr s25, [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v9.h }[2], [x13]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "ld1 { v21.h }[2], [x24]\n"
      "ld1 { v25.h }[2], [x23]\n"
      "b 220f\n"
      "215:"  // Height 5: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x14, #0, 220f\n"
      "ldr h9, [x13, #0x0]\n"
      "ldr h13, [x26, #0x0]\n"
      "ldr h17, [x25, #0x0]\n"
      "ldr h21, [x24, #0x0]\n"
      "ldr h25, [x23, #0x0]\n"
      "b 220f\n"
      "216:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x14, #2, 218f\n"
      "ldr d8, [x13], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "tbz x14, #1, 217f\n"
      "ld1 { v8.s }[2], [x13], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "ld1 { v20.s }[2], [x24], #0x4\n"
      "ld1 { v24.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v8.h }[6], [x13]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "ld1 { v20.h }[6], [x24]\n"
      "ld1 { v24.h }[6], [x23]\n"
      "b 220f\n"
      "217:"  // Height 5: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v8.h }[4], [x13]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "ld1 { v20.h }[4], [x24]\n"
      "ld1 { v24.h }[4], [x23]\n"
      "b 220f\n"
      "218:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x14, #1, 219f\n"
      "ldr s8, [x13], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "ldr s24, [x23], #0x4\n"
      "tbz x14, #0, 220f\n"
      "ld1 { v8.h }[2], [x13]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "ld1 { v20.h }[2], [x24]\n"
      "ld1 { v24.h }[2], [x23]\n"
      "b 220f\n"
      "219:"  // Height 5: Partial accumulate: partial_1_0
      "ldr h8, [x13, #0x0]\n"
      "ldr h12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr h16, [x25, #0x0]\n"
      "ldr h20, [x24, #0x0]\n"
      "ldr h24, [x23, #0x0]\n"
      "220:"  // Height 5: Partial accumulate: Done
      "sub x13, x13, x20\n"
      "b 223f\n"
      "221:"  // Height 5: full accumulate
      "ldr q8, [x13, #0x0]\n"
      "ldr q9, [x13, #0x10]\n"
      "ldr q10, [x13, #0x20]\n"
      "ldr q11, [x13, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0x0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "ldr q20, [x24, #0x0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "ldr q24, [x23, #0x0]\n"
      "ldr q25, [x23, #0x10]\n"
      "ldr q26, [x23, #0x20]\n"
      "ldr q27, [x23, #0x30]\n"
      "b 223f\n"
      "222:"  // Height 5: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "223:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "224:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 225f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 226f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "b 226f\n"
      "225:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "226:"  // Height 5: input setup done
      "cmp x27, #0x8\n"
      "blt 229f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q6, [x12, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "blt 228f\n"
      "227:"  // Height 5: Multiply loop: Main loop head
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "fmla v20.8h, v6.8h, v3.h[0]\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v24.8h, v6.8h, v4.h[0]\n"
      "ldr q29, [x10, #0x0]\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "add x24, x24, #0x10\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "fmla v21.8h, v7.8h, v3.h[0]\n"
      "fmla v25.8h, v7.8h, v4.h[0]\n"
      "ldr q28, [x9, #0x0]\n"
      "fmla v10.8h, v29.8h, v0.h[0]\n"
      "fmla v14.8h, v29.8h, v1.h[0]\n"
      "fmla v18.8h, v29.8h, v2.h[0]\n"
      "fmla v22.8h, v29.8h, v3.h[0]\n"
      "fmla v26.8h, v29.8h, v4.h[0]\n"
      "ldr q29, [x12, #0x10]\n"
      "fmla v11.8h, v28.8h, v0.h[0]\n"
      "fmla v15.8h, v28.8h, v1.h[0]\n"
      "fmla v19.8h, v28.8h, v2.h[0]\n"
      "fmla v23.8h, v28.8h, v3.h[0]\n"
      "fmla v27.8h, v28.8h, v4.h[0]\n"
      "ldr q28, [x11, #0x10]\n"
      "fmla v8.8h, v29.8h, v0.h[1]\n"
      "fmla v12.8h, v29.8h, v1.h[1]\n"
      "fmla v16.8h, v29.8h, v2.h[1]\n"
      "fmla v20.8h, v29.8h, v3.h[1]\n"
      "fmla v24.8h, v29.8h, v4.h[1]\n"
      "ldr q29, [x10, #0x10]\n"
      "fmla v9.8h, v28.8h, v0.h[1]\n"
      "fmla v13.8h, v28.8h, v1.h[1]\n"
      "fmla v17.8h, v28.8h, v2.h[1]\n"
      "fmla v21.8h, v28.8h, v3.h[1]\n"
      "fmla v25.8h, v28.8h, v4.h[1]\n"
      "ldr q28, [x9, #0x10]\n"
      "fmla v10.8h, v29.8h, v0.h[1]\n"
      "fmla v14.8h, v29.8h, v1.h[1]\n"
      "fmla v18.8h, v29.8h, v2.h[1]\n"
      "fmla v22.8h, v29.8h, v3.h[1]\n"
      "fmla v26.8h, v29.8h, v4.h[1]\n"
      "ldr q29, [x12, #0x20]\n"
      "fmla v11.8h, v28.8h, v0.h[1]\n"
      "fmla v15.8h, v28.8h, v1.h[1]\n"
      "fmla v19.8h, v28.8h, v2.h[1]\n"
      "fmla v23.8h, v28.8h, v3.h[1]\n"
      "fmla v27.8h, v28.8h, v4.h[1]\n"
      "ldr q28, [x11, #0x20]\n"
      "fmla v8.8h, v29.8h, v0.h[2]\n"
      "fmla v12.8h, v29.8h, v1.h[2]\n"
      "fmla v16.8h, v29.8h, v2.h[2]\n"
      "fmla v20.8h, v29.8h, v3.h[2]\n"
      "fmla v24.8h, v29.8h, v4.h[2]\n"
      "ldr q29, [x10, #0x20]\n"
      "fmla v9.8h, v28.8h, v0.h[2]\n"
      "fmla v13.8h, v28.8h, v1.h[2]\n"
      "fmla v17.8h, v28.8h, v2.h[2]\n"
      "fmla v21.8h, v28.8h, v3.h[2]\n"
      "fmla v25.8h, v28.8h, v4.h[2]\n"
      "ldr q28, [x9, #0x20]\n"
      "fmla v10.8h, v29.8h, v0.h[2]\n"
      "fmla v14.8h, v29.8h, v1.h[2]\n"
      "fmla v18.8h, v29.8h, v2.h[2]\n"
      "fmla v22.8h, v29.8h, v3.h[2]\n"
      "fmla v26.8h, v29.8h, v4.h[2]\n"
      "ldr q29, [x12, #0x30]\n"
      "fmla v11.8h, v28.8h, v0.h[2]\n"
      "fmla v15.8h, v28.8h, v1.h[2]\n"
      "fmla v19.8h, v28.8h, v2.h[2]\n"
      "fmla v23.8h, v28.8h, v3.h[2]\n"
      "fmla v27.8h, v28.8h, v4.h[2]\n"
      "ldr q28, [x11, #0x30]\n"
      "fmla v8.8h, v29.8h, v0.h[3]\n"
      "fmla v12.8h, v29.8h, v1.h[3]\n"
      "fmla v16.8h, v29.8h, v2.h[3]\n"
      "fmla v20.8h, v29.8h, v3.h[3]\n"
      "fmla v24.8h, v29.8h, v4.h[3]\n"
      "ldr q29, [x10, #0x30]\n"
      "fmla v9.8h, v28.8h, v0.h[3]\n"
      "fmla v13.8h, v28.8h, v1.h[3]\n"
      "fmla v17.8h, v28.8h, v2.h[3]\n"
      "fmla v21.8h, v28.8h, v3.h[3]\n"
      "fmla v25.8h, v28.8h, v4.h[3]\n"
      "ldr q28, [x9, #0x30]\n"
      "fmla v10.8h, v29.8h, v0.h[3]\n"
      "fmla v14.8h, v29.8h, v1.h[3]\n"
      "fmla v18.8h, v29.8h, v2.h[3]\n"
      "fmla v22.8h, v29.8h, v3.h[3]\n"
      "fmla v26.8h, v29.8h, v4.h[3]\n"
      "ldr q29, [x12, #0x40]\n"
      "fmla v11.8h, v28.8h, v0.h[3]\n"
      "fmla v15.8h, v28.8h, v1.h[3]\n"
      "fmla v19.8h, v28.8h, v2.h[3]\n"
      "fmla v23.8h, v28.8h, v3.h[3]\n"
      "fmla v27.8h, v28.8h, v4.h[3]\n"
      "ldr q28, [x11, #0x40]\n"
      "fmla v8.8h, v29.8h, v0.h[4]\n"
      "fmla v12.8h, v29.8h, v1.h[4]\n"
      "fmla v16.8h, v29.8h, v2.h[4]\n"
      "fmla v20.8h, v29.8h, v3.h[4]\n"
      "fmla v24.8h, v29.8h, v4.h[4]\n"
      "ldr q29, [x10, #0x40]\n"
      "fmla v9.8h, v28.8h, v0.h[4]\n"
      "fmla v13.8h, v28.8h, v1.h[4]\n"
      "fmla v17.8h, v28.8h, v2.h[4]\n"
      "fmla v21.8h, v28.8h, v3.h[4]\n"
      "fmla v25.8h, v28.8h, v4.h[4]\n"
      "ldr q28, [x9, #0x40]\n"
      "fmla v10.8h, v29.8h, v0.h[4]\n"
      "fmla v14.8h, v29.8h, v1.h[4]\n"
      "fmla v18.8h, v29.8h, v2.h[4]\n"
      "fmla v22.8h, v29.8h, v3.h[4]\n"
      "fmla v26.8h, v29.8h, v4.h[4]\n"
      "ldr q29, [x12, #0x50]\n"
      "fmla v11.8h, v28.8h, v0.h[4]\n"
      "fmla v15.8h, v28.8h, v1.h[4]\n"
      "fmla v19.8h, v28.8h, v2.h[4]\n"
      "fmla v23.8h, v28.8h, v3.h[4]\n"
      "fmla v27.8h, v28.8h, v4.h[4]\n"
      "ldr q28, [x11, #0x50]\n"
      "fmla v8.8h, v29.8h, v0.h[5]\n"
      "fmla v12.8h, v29.8h, v1.h[5]\n"
      "fmla v16.8h, v29.8h, v2.h[5]\n"
      "fmla v20.8h, v29.8h, v3.h[5]\n"
      "fmla v24.8h, v29.8h, v4.h[5]\n"
      "ldr q29, [x10, #0x50]\n"
      "fmla v9.8h, v28.8h, v0.h[5]\n"
      "fmla v13.8h, v28.8h, v1.h[5]\n"
      "fmla v17.8h, v28.8h, v2.h[5]\n"
      "fmla v21.8h, v28.8h, v3.h[5]\n"
      "fmla v25.8h, v28.8h, v4.h[5]\n"
      "ldr q28, [x9, #0x50]\n"
      "fmla v10.8h, v29.8h, v0.h[5]\n"
      "fmla v14.8h, v29.8h, v1.h[5]\n"
      "fmla v18.8h, v29.8h, v2.h[5]\n"
      "fmla v22.8h, v29.8h, v3.h[5]\n"
      "fmla v26.8h, v29.8h, v4.h[5]\n"
      "ldr q29, [x12, #0x60]\n"
      "fmla v11.8h, v28.8h, v0.h[5]\n"
      "fmla v15.8h, v28.8h, v1.h[5]\n"
      "fmla v19.8h, v28.8h, v2.h[5]\n"
      "fmla v23.8h, v28.8h, v3.h[5]\n"
      "fmla v27.8h, v28.8h, v4.h[5]\n"
      "ldr q28, [x11, #0x60]\n"
      "fmla v8.8h, v29.8h, v0.h[6]\n"
      "fmla v12.8h, v29.8h, v1.h[6]\n"
      "fmla v16.8h, v29.8h, v2.h[6]\n"
      "fmla v20.8h, v29.8h, v3.h[6]\n"
      "fmla v24.8h, v29.8h, v4.h[6]\n"
      "ldr q29, [x10, #0x60]\n"
      "fmla v9.8h, v28.8h, v0.h[6]\n"
      "fmla v13.8h, v28.8h, v1.h[6]\n"
      "fmla v17.8h, v28.8h, v2.h[6]\n"
      "fmla v21.8h, v28.8h, v3.h[6]\n"
      "fmla v25.8h, v28.8h, v4.h[6]\n"
      "ldr q28, [x9, #0x60]\n"
      "fmla v10.8h, v29.8h, v0.h[6]\n"
      "fmla v14.8h, v29.8h, v1.h[6]\n"
      "fmla v18.8h, v29.8h, v2.h[6]\n"
      "fmla v22.8h, v29.8h, v3.h[6]\n"
      "fmla v26.8h, v29.8h, v4.h[6]\n"
      "ldr q29, [x12, #0x70]\n"
      "fmla v11.8h, v28.8h, v0.h[6]\n"
      "add x12, x12, #0x80\n"
      "fmla v15.8h, v28.8h, v1.h[6]\n"
      "fmla v19.8h, v28.8h, v2.h[6]\n"
      "fmla v23.8h, v28.8h, v3.h[6]\n"
      "fmla v27.8h, v28.8h, v4.h[6]\n"
      "ldr q28, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v29.8h, v0.h[7]\n"
      "fmla v12.8h, v29.8h, v1.h[7]\n"
      "fmla v16.8h, v29.8h, v2.h[7]\n"
      "fmla v20.8h, v29.8h, v3.h[7]\n"
      "fmla v24.8h, v29.8h, v4.h[7]\n"
      "ldr q29, [x10, #0x70]\n"
      "fmla v9.8h, v28.8h, v0.h[7]\n"
      "add x10, x10, #0x80\n"
      "fmla v13.8h, v28.8h, v1.h[7]\n"
      "fmla v17.8h, v28.8h, v2.h[7]\n"
      "fmla v21.8h, v28.8h, v3.h[7]\n"
      "fmla v25.8h, v28.8h, v4.h[7]\n"
      "ldr q28, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v29.8h, v0.h[7]\n"
      "fmla v14.8h, v29.8h, v1.h[7]\n"
      "fmla v18.8h, v29.8h, v2.h[7]\n"
      "fmla v22.8h, v29.8h, v3.h[7]\n"
      "fmla v26.8h, v29.8h, v4.h[7]\n"
      "ldr q6, [x12, #0x0]\n"
      "fmla v11.8h, v28.8h, v0.h[7]\n"
      "ldr q0, [x26, #0x0]\n"
      "fmla v15.8h, v28.8h, v1.h[7]\n"
      "ldr q1, [x25, #0x0]\n"
      "fmla v19.8h, v28.8h, v2.h[7]\n"
      "ldr q2, [x24, #0x0]\n"
      "fmla v23.8h, v28.8h, v3.h[7]\n"
      "ldr q3, [x23, #0x0]\n"
      "fmla v27.8h, v28.8h, v4.h[7]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "bge 227b\n"
      "228:"  // Height 5: Multiply loop: Single iteration only
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "fmla v20.8h, v6.8h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v24.8h, v6.8h, v4.h[0]\n"
      "ldr q29, [x10, #0x0]\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "add x23, x23, #0x10\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "add x22, x22, #0x10\n"
      "fmla v21.8h, v7.8h, v3.h[0]\n"
      "fmla v25.8h, v7.8h, v4.h[0]\n"
      "ldr q28, [x9, #0x0]\n"
      "fmla v10.8h, v29.8h, v0.h[0]\n"
      "fmla v14.8h, v29.8h, v1.h[0]\n"
      "fmla v18.8h, v29.8h, v2.h[0]\n"
      "fmla v22.8h, v29.8h, v3.h[0]\n"
      "fmla v26.8h, v29.8h, v4.h[0]\n"
      "ldr q29, [x12, #0x10]\n"
      "fmla v11.8h, v28.8h, v0.h[0]\n"
      "fmla v15.8h, v28.8h, v1.h[0]\n"
      "fmla v19.8h, v28.8h, v2.h[0]\n"
      "fmla v23.8h, v28.8h, v3.h[0]\n"
      "fmla v27.8h, v28.8h, v4.h[0]\n"
      "ldr q28, [x11, #0x10]\n"
      "fmla v8.8h, v29.8h, v0.h[1]\n"
      "fmla v12.8h, v29.8h, v1.h[1]\n"
      "fmla v16.8h, v29.8h, v2.h[1]\n"
      "fmla v20.8h, v29.8h, v3.h[1]\n"
      "fmla v24.8h, v29.8h, v4.h[1]\n"
      "ldr q29, [x10, #0x10]\n"
      "fmla v9.8h, v28.8h, v0.h[1]\n"
      "fmla v13.8h, v28.8h, v1.h[1]\n"
      "fmla v17.8h, v28.8h, v2.h[1]\n"
      "fmla v21.8h, v28.8h, v3.h[1]\n"
      "fmla v25.8h, v28.8h, v4.h[1]\n"
      "ldr q28, [x9, #0x10]\n"
      "fmla v10.8h, v29.8h, v0.h[1]\n"
      "fmla v14.8h, v29.8h, v1.h[1]\n"
      "fmla v18.8h, v29.8h, v2.h[1]\n"
      "fmla v22.8h, v29.8h, v3.h[1]\n"
      "fmla v26.8h, v29.8h, v4.h[1]\n"
      "ldr q29, [x12, #0x20]\n"
      "fmla v11.8h, v28.8h, v0.h[1]\n"
      "fmla v15.8h, v28.8h, v1.h[1]\n"
      "fmla v19.8h, v28.8h, v2.h[1]\n"
      "fmla v23.8h, v28.8h, v3.h[1]\n"
      "fmla v27.8h, v28.8h, v4.h[1]\n"
      "ldr q28, [x11, #0x20]\n"
      "fmla v8.8h, v29.8h, v0.h[2]\n"
      "fmla v12.8h, v29.8h, v1.h[2]\n"
      "fmla v16.8h, v29.8h, v2.h[2]\n"
      "fmla v20.8h, v29.8h, v3.h[2]\n"
      "fmla v24.8h, v29.8h, v4.h[2]\n"
      "ldr q29, [x10, #0x20]\n"
      "fmla v9.8h, v28.8h, v0.h[2]\n"
      "fmla v13.8h, v28.8h, v1.h[2]\n"
      "fmla v17.8h, v28.8h, v2.h[2]\n"
      "fmla v21.8h, v28.8h, v3.h[2]\n"
      "fmla v25.8h, v28.8h, v4.h[2]\n"
      "ldr q28, [x9, #0x20]\n"
      "fmla v10.8h, v29.8h, v0.h[2]\n"
      "fmla v14.8h, v29.8h, v1.h[2]\n"
      "fmla v18.8h, v29.8h, v2.h[2]\n"
      "fmla v22.8h, v29.8h, v3.h[2]\n"
      "fmla v26.8h, v29.8h, v4.h[2]\n"
      "ldr q29, [x12, #0x30]\n"
      "fmla v11.8h, v28.8h, v0.h[2]\n"
      "fmla v15.8h, v28.8h, v1.h[2]\n"
      "fmla v19.8h, v28.8h, v2.h[2]\n"
      "fmla v23.8h, v28.8h, v3.h[2]\n"
      "fmla v27.8h, v28.8h, v4.h[2]\n"
      "ldr q28, [x11, #0x30]\n"
      "fmla v8.8h, v29.8h, v0.h[3]\n"
      "fmla v12.8h, v29.8h, v1.h[3]\n"
      "fmla v16.8h, v29.8h, v2.h[3]\n"
      "fmla v20.8h, v29.8h, v3.h[3]\n"
      "fmla v24.8h, v29.8h, v4.h[3]\n"
      "ldr q29, [x10, #0x30]\n"
      "fmla v9.8h, v28.8h, v0.h[3]\n"
      "fmla v13.8h, v28.8h, v1.h[3]\n"
      "fmla v17.8h, v28.8h, v2.h[3]\n"
      "fmla v21.8h, v28.8h, v3.h[3]\n"
      "fmla v25.8h, v28.8h, v4.h[3]\n"
      "ldr q28, [x9, #0x30]\n"
      "fmla v10.8h, v29.8h, v0.h[3]\n"
      "fmla v14.8h, v29.8h, v1.h[3]\n"
      "fmla v18.8h, v29.8h, v2.h[3]\n"
      "fmla v22.8h, v29.8h, v3.h[3]\n"
      "fmla v26.8h, v29.8h, v4.h[3]\n"
      "ldr q29, [x12, #0x40]\n"
      "fmla v11.8h, v28.8h, v0.h[3]\n"
      "fmla v15.8h, v28.8h, v1.h[3]\n"
      "fmla v19.8h, v28.8h, v2.h[3]\n"
      "fmla v23.8h, v28.8h, v3.h[3]\n"
      "fmla v27.8h, v28.8h, v4.h[3]\n"
      "ldr q28, [x11, #0x40]\n"
      "fmla v8.8h, v29.8h, v0.h[4]\n"
      "fmla v12.8h, v29.8h, v1.h[4]\n"
      "fmla v16.8h, v29.8h, v2.h[4]\n"
      "fmla v20.8h, v29.8h, v3.h[4]\n"
      "fmla v24.8h, v29.8h, v4.h[4]\n"
      "ldr q29, [x10, #0x40]\n"
      "fmla v9.8h, v28.8h, v0.h[4]\n"
      "fmla v13.8h, v28.8h, v1.h[4]\n"
      "fmla v17.8h, v28.8h, v2.h[4]\n"
      "fmla v21.8h, v28.8h, v3.h[4]\n"
      "fmla v25.8h, v28.8h, v4.h[4]\n"
      "ldr q28, [x9, #0x40]\n"
      "fmla v10.8h, v29.8h, v0.h[4]\n"
      "fmla v14.8h, v29.8h, v1.h[4]\n"
      "fmla v18.8h, v29.8h, v2.h[4]\n"
      "fmla v22.8h, v29.8h, v3.h[4]\n"
      "fmla v26.8h, v29.8h, v4.h[4]\n"
      "ldr q29, [x12, #0x50]\n"
      "fmla v11.8h, v28.8h, v0.h[4]\n"
      "fmla v15.8h, v28.8h, v1.h[4]\n"
      "fmla v19.8h, v28.8h, v2.h[4]\n"
      "fmla v23.8h, v28.8h, v3.h[4]\n"
      "fmla v27.8h, v28.8h, v4.h[4]\n"
      "ldr q28, [x11, #0x50]\n"
      "fmla v8.8h, v29.8h, v0.h[5]\n"
      "fmla v12.8h, v29.8h, v1.h[5]\n"
      "fmla v16.8h, v29.8h, v2.h[5]\n"
      "fmla v20.8h, v29.8h, v3.h[5]\n"
      "fmla v24.8h, v29.8h, v4.h[5]\n"
      "ldr q29, [x10, #0x50]\n"
      "fmla v9.8h, v28.8h, v0.h[5]\n"
      "fmla v13.8h, v28.8h, v1.h[5]\n"
      "fmla v17.8h, v28.8h, v2.h[5]\n"
      "fmla v21.8h, v28.8h, v3.h[5]\n"
      "fmla v25.8h, v28.8h, v4.h[5]\n"
      "ldr q28, [x9, #0x50]\n"
      "fmla v10.8h, v29.8h, v0.h[5]\n"
      "fmla v14.8h, v29.8h, v1.h[5]\n"
      "fmla v18.8h, v29.8h, v2.h[5]\n"
      "fmla v22.8h, v29.8h, v3.h[5]\n"
      "fmla v26.8h, v29.8h, v4.h[5]\n"
      "ldr q29, [x12, #0x60]\n"
      "fmla v11.8h, v28.8h, v0.h[5]\n"
      "fmla v15.8h, v28.8h, v1.h[5]\n"
      "fmla v19.8h, v28.8h, v2.h[5]\n"
      "fmla v23.8h, v28.8h, v3.h[5]\n"
      "fmla v27.8h, v28.8h, v4.h[5]\n"
      "ldr q28, [x11, #0x60]\n"
      "fmla v8.8h, v29.8h, v0.h[6]\n"
      "fmla v12.8h, v29.8h, v1.h[6]\n"
      "fmla v16.8h, v29.8h, v2.h[6]\n"
      "fmla v20.8h, v29.8h, v3.h[6]\n"
      "fmla v24.8h, v29.8h, v4.h[6]\n"
      "ldr q29, [x10, #0x60]\n"
      "fmla v9.8h, v28.8h, v0.h[6]\n"
      "fmla v13.8h, v28.8h, v1.h[6]\n"
      "fmla v17.8h, v28.8h, v2.h[6]\n"
      "fmla v21.8h, v28.8h, v3.h[6]\n"
      "fmla v25.8h, v28.8h, v4.h[6]\n"
      "ldr q28, [x9, #0x60]\n"
      "fmla v10.8h, v29.8h, v0.h[6]\n"
      "fmla v14.8h, v29.8h, v1.h[6]\n"
      "fmla v18.8h, v29.8h, v2.h[6]\n"
      "fmla v22.8h, v29.8h, v3.h[6]\n"
      "fmla v26.8h, v29.8h, v4.h[6]\n"
      "ldr q29, [x12, #0x70]\n"
      "fmla v11.8h, v28.8h, v0.h[6]\n"
      "add x12, x12, #0x80\n"
      "fmla v15.8h, v28.8h, v1.h[6]\n"
      "fmla v19.8h, v28.8h, v2.h[6]\n"
      "fmla v23.8h, v28.8h, v3.h[6]\n"
      "fmla v27.8h, v28.8h, v4.h[6]\n"
      "ldr q28, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v29.8h, v0.h[7]\n"
      "fmla v12.8h, v29.8h, v1.h[7]\n"
      "fmla v16.8h, v29.8h, v2.h[7]\n"
      "fmla v20.8h, v29.8h, v3.h[7]\n"
      "fmla v24.8h, v29.8h, v4.h[7]\n"
      "ldr q29, [x10, #0x70]\n"
      "fmla v9.8h, v28.8h, v0.h[7]\n"
      "add x10, x10, #0x80\n"
      "fmla v13.8h, v28.8h, v1.h[7]\n"
      "fmla v17.8h, v28.8h, v2.h[7]\n"
      "fmla v21.8h, v28.8h, v3.h[7]\n"
      "fmla v25.8h, v28.8h, v4.h[7]\n"
      "ldr q28, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v29.8h, v0.h[7]\n"
      "fmla v14.8h, v29.8h, v1.h[7]\n"
      "fmla v18.8h, v29.8h, v2.h[7]\n"
      "fmla v22.8h, v29.8h, v3.h[7]\n"
      "fmla v26.8h, v29.8h, v4.h[7]\n"
      "fmla v11.8h, v28.8h, v0.h[7]\n"
      "fmla v15.8h, v28.8h, v1.h[7]\n"
      "fmla v19.8h, v28.8h, v2.h[7]\n"
      "fmla v23.8h, v28.8h, v3.h[7]\n"
      "fmla v27.8h, v28.8h, v4.h[7]\n"
      "229:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 231f\n"
      "230:"  // Height 5: Multiply loop: Odd block loop
      "ldr h4, [x26], #0x2\n"
      "ldr h3, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h1, [x23], #0x2\n"
      "ldr h0, [x22], #0x2\n"
      "ldr q29, [x12, #0x0]\n"
      "add x12, x12, #0x10\n"
      "ldr q28, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "fmla v8.8h, v29.8h, v4.h[0]\n"
      "fmla v12.8h, v29.8h, v3.h[0]\n"
      "fmla v16.8h, v29.8h, v2.h[0]\n"
      "fmla v20.8h, v29.8h, v1.h[0]\n"
      "fmla v24.8h, v29.8h, v0.h[0]\n"
      "ldr q29, [x10, #0x0]\n"
      "add x10, x10, #0x10\n"
      "fmla v9.8h, v28.8h, v4.h[0]\n"
      "fmla v13.8h, v28.8h, v3.h[0]\n"
      "fmla v17.8h, v28.8h, v2.h[0]\n"
      "fmla v21.8h, v28.8h, v1.h[0]\n"
      "fmla v25.8h, v28.8h, v0.h[0]\n"
      "ldr q28, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "fmla v10.8h, v29.8h, v4.h[0]\n"
      "fmla v14.8h, v29.8h, v3.h[0]\n"
      "fmla v18.8h, v29.8h, v2.h[0]\n"
      "fmla v22.8h, v29.8h, v1.h[0]\n"
      "fmla v26.8h, v29.8h, v0.h[0]\n"
      "fmla v11.8h, v28.8h, v4.h[0]\n"
      "fmla v15.8h, v28.8h, v3.h[0]\n"
      "fmla v19.8h, v28.8h, v2.h[0]\n"
      "fmla v23.8h, v28.8h, v1.h[0]\n"
      "fmla v27.8h, v28.8h, v0.h[0]\n"
      "cbnz x27, 230b\n"
      "231:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 224b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "tbz %x[flags], #1, 232f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v29.8h }, [x21]\n"
      "ld1r { v28.8h }, [x20]\n"
      "fmin v8.8h, v8.8h, v29.8h\n"
      "fmin v9.8h, v9.8h, v29.8h\n"
      "fmin v10.8h, v10.8h, v29.8h\n"
      "fmin v11.8h, v11.8h, v29.8h\n"
      "fmin v12.8h, v12.8h, v29.8h\n"
      "fmin v13.8h, v13.8h, v29.8h\n"
      "fmin v14.8h, v14.8h, v29.8h\n"
      "fmin v15.8h, v15.8h, v29.8h\n"
      "fmin v16.8h, v16.8h, v29.8h\n"
      "fmin v17.8h, v17.8h, v29.8h\n"
      "fmin v18.8h, v18.8h, v29.8h\n"
      "fmin v19.8h, v19.8h, v29.8h\n"
      "fmin v20.8h, v20.8h, v29.8h\n"
      "fmin v21.8h, v21.8h, v29.8h\n"
      "fmin v22.8h, v22.8h, v29.8h\n"
      "fmin v23.8h, v23.8h, v29.8h\n"
      "fmin v24.8h, v24.8h, v29.8h\n"
      "fmin v25.8h, v25.8h, v29.8h\n"
      "fmin v26.8h, v26.8h, v29.8h\n"
      "fmin v27.8h, v27.8h, v29.8h\n"
      "fmax v8.8h, v8.8h, v28.8h\n"
      "fmax v9.8h, v9.8h, v28.8h\n"
      "fmax v10.8h, v10.8h, v28.8h\n"
      "fmax v11.8h, v11.8h, v28.8h\n"
      "fmax v12.8h, v12.8h, v28.8h\n"
      "fmax v13.8h, v13.8h, v28.8h\n"
      "fmax v14.8h, v14.8h, v28.8h\n"
      "fmax v15.8h, v15.8h, v28.8h\n"
      "fmax v16.8h, v16.8h, v28.8h\n"
      "fmax v17.8h, v17.8h, v28.8h\n"
      "fmax v18.8h, v18.8h, v28.8h\n"
      "fmax v19.8h, v19.8h, v28.8h\n"
      "fmax v20.8h, v20.8h, v28.8h\n"
      "fmax v21.8h, v21.8h, v28.8h\n"
      "fmax v22.8h, v22.8h, v28.8h\n"
      "fmax v23.8h, v23.8h, v28.8h\n"
      "fmax v24.8h, v24.8h, v28.8h\n"
      "fmax v25.8h, v25.8h, v28.8h\n"
      "fmax v26.8h, v26.8h, v28.8h\n"
      "fmax v27.8h, v27.8h, v28.8h\n"
      "232:"  // Height 5: No activation
      "cmp x14, #0x20\n"
      "bge 249f\n"
      "tbz x14, #4, 240f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v9.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v13.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v17.8h }, [x25], #0x10\n"
      "st1 { v20.8h }, [x24], #0x10\n"
      "st1 { v21.8h }, [x24], #0x10\n"
      "st1 { v24.8h }, [x23], #0x10\n"
      "st1 { v25.8h }, [x23], #0x10\n"
      "tbz x14, #3, 236f\n"
      "st1 { v10.8h }, [x13], #0x10\n"
      "st1 { v14.8h }, [x26], #0x10\n"
      "st1 { v18.8h }, [x25], #0x10\n"
      "st1 { v22.8h }, [x24], #0x10\n"
      "st1 { v26.8h }, [x23], #0x10\n"
      "tbz x14, #2, 234f\n"
      "str d11, [x13], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "str d27, [x23], #0x8\n"
      "tbz x14, #1, 233f\n"
      "st1 { v11.s }[2], [x13], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v19.s }[2], [x25], #0x4\n"
      "st1 { v23.s }[2], [x24], #0x4\n"
      "st1 { v27.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v11.h }[6], [x13]\n"
      "st1 { v15.h }[6], [x26]\n"
      "st1 { v19.h }[6], [x25]\n"
      "st1 { v23.h }[6], [x24]\n"
      "st1 { v27.h }[6], [x23]\n"
      "b 248f\n"
      "233:"  // Height 5: Partial direct writeback: partial_1_28
      "tbz x14, #0, 248f\n"
      "st1 { v11.h }[4], [x13]\n"
      "st1 { v15.h }[4], [x26]\n"
      "st1 { v19.h }[4], [x25]\n"
      "st1 { v23.h }[4], [x24]\n"
      "st1 { v27.h }[4], [x23]\n"
      "b 248f\n"
      "234:"  // Height 5: Partial direct writeback: partial_2_24
      "tbz x14, #1, 235f\n"
      "str s11, [x13], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s19, [x25], #0x4\n"
      "str s23, [x24], #0x4\n"
      "str s27, [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v11.h }[2], [x13]\n"
      "st1 { v15.h }[2], [x26]\n"
      "st1 { v19.h }[2], [x25]\n"
      "st1 { v23.h }[2], [x24]\n"
      "st1 { v27.h }[2], [x23]\n"
      "b 248f\n"
      "235:"  // Height 5: Partial direct writeback: partial_1_24
      "tbz x14, #0, 248f\n"
      "str h11, [x13, #0x0]\n"
      "str h15, [x26, #0x0]\n"
      "str h19, [x25, #0x0]\n"
      "str h23, [x24, #0x0]\n"
      "str h27, [x23, #0x0]\n"
      "b 248f\n"
      "236:"  // Height 5: Partial direct writeback: partial_4_16
      "tbz x14, #2, 238f\n"
      "str d10, [x13], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "str d22, [x24], #0x8\n"
      "str d26, [x23], #0x8\n"
      "tbz x14, #1, 237f\n"
      "st1 { v10.s }[2], [x13], #0x4\n"
      "st1 { v14.s }[2], [x26], #0x4\n"
      "st1 { v18.s }[2], [x25], #0x4\n"
      "st1 { v22.s }[2], [x24], #0x4\n"
      "st1 { v26.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v10.h }[6], [x13]\n"
      "st1 { v14.h }[6], [x26]\n"
      "st1 { v18.h }[6], [x25]\n"
      "st1 { v22.h }[6], [x24]\n"
      "st1 { v26.h }[6], [x23]\n"
      "b 248f\n"
      "237:"  // Height 5: Partial direct writeback: partial_1_20
      "tbz x14, #0, 248f\n"
      "st1 { v10.h }[4], [x13]\n"
      "st1 { v14.h }[4], [x26]\n"
      "st1 { v18.h }[4], [x25]\n"
      "st1 { v22.h }[4], [x24]\n"
      "st1 { v26.h }[4], [x23]\n"
      "b 248f\n"
      "238:"  // Height 5: Partial direct writeback: partial_2_16
      "tbz x14, #1, 239f\n"
      "str s10, [x13], #0x4\n"
      "str s14, [x26], #0x4\n"
      "str s18, [x25], #0x4\n"
      "str s22, [x24], #0x4\n"
      "str s26, [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v10.h }[2], [x13]\n"
      "st1 { v14.h }[2], [x26]\n"
      "st1 { v18.h }[2], [x25]\n"
      "st1 { v22.h }[2], [x24]\n"
      "st1 { v26.h }[2], [x23]\n"
      "b 248f\n"
      "239:"  // Height 5: Partial direct writeback: partial_1_16
      "tbz x14, #0, 248f\n"
      "str h10, [x13, #0x0]\n"
      "str h14, [x26, #0x0]\n"
      "str h18, [x25, #0x0]\n"
      "str h22, [x24, #0x0]\n"
      "str h26, [x23, #0x0]\n"
      "b 248f\n"
      "240:"  // Height 5: Partial direct writeback: partial_8_0
      "tbz x14, #3, 244f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v20.8h }, [x24], #0x10\n"
      "st1 { v24.8h }, [x23], #0x10\n"
      "tbz x14, #2, 242f\n"
      "str d9, [x13], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "tbz x14, #1, 241f\n"
      "st1 { v9.s }[2], [x13], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "st1 { v21.s }[2], [x24], #0x4\n"
      "st1 { v25.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v9.h }[6], [x13]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "st1 { v21.h }[6], [x24]\n"
      "st1 { v25.h }[6], [x23]\n"
      "b 248f\n"
      "241:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x14, #0, 248f\n"
      "st1 { v9.h }[4], [x13]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "st1 { v21.h }[4], [x24]\n"
      "st1 { v25.h }[4], [x23]\n"
      "b 248f\n"
      "242:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x14, #1, 243f\n"
      "str s9, [x13], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "str s21, [x24], #0x4\n"
      "str s25, [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v9.h }[2], [x13]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "st1 { v21.h }[2], [x24]\n"
      "st1 { v25.h }[2], [x23]\n"
      "b 248f\n"
      "243:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x14, #0, 248f\n"
      "str h9, [x13, #0x0]\n"
      "str h13, [x26, #0x0]\n"
      "str h17, [x25, #0x0]\n"
      "str h21, [x24, #0x0]\n"
      "str h25, [x23, #0x0]\n"
      "b 248f\n"
      "244:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x14, #2, 246f\n"
      "str d8, [x13], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x14, #1, 245f\n"
      "st1 { v8.s }[2], [x13], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v8.h }[6], [x13]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "st1 { v20.h }[6], [x24]\n"
      "st1 { v24.h }[6], [x23]\n"
      "b 248f\n"
      "245:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x14, #0, 248f\n"
      "st1 { v8.h }[4], [x13]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "st1 { v20.h }[4], [x24]\n"
      "st1 { v24.h }[4], [x23]\n"
      "b 248f\n"
      "246:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x14, #1, 247f\n"
      "str s8, [x13], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "tbz x14, #0, 248f\n"
      "st1 { v8.h }[2], [x13]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "st1 { v20.h }[2], [x24]\n"
      "st1 { v24.h }[2], [x23]\n"
      "b 248f\n"
      "247:"  // Height 5: Partial direct writeback: partial_1_0
      "str h8, [x13, #0x0]\n"
      "str h12, [x26, #0x0]\n"
      "str h16, [x25, #0x0]\n"
      "str h20, [x24, #0x0]\n"
      "str h24, [x23, #0x0]\n"
      "248:"  // Height 5: Partial direct writeback: Done
      "b 250f\n"
      "249:"  // Height 5: Full writeback
      "str q8, [x13, #0x0]\n"
      "str q9, [x13, #0x10]\n"
      "str q10, [x13, #0x20]\n"
      "str q11, [x13, #0x30]\n"
      "add x13, x13, #0x40\n"
      "str q12, [x26, #0x0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0x0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "str q20, [x24, #0x0]\n"
      "str q21, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "str q23, [x24, #0x30]\n"
      "str q24, [x23, #0x0]\n"
      "str q25, [x23, #0x10]\n"
      "str q26, [x23, #0x20]\n"
      "str q27, [x23, #0x30]\n"
      "250:"  // Height 5: Writeback done
      "subs x14, x14, #0x20\n"
      "bgt 202b\n"
      "b 302f\n"
      "251:"  // Height 6
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "mov x21, #0xc\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "madd x21, x20, x21, x13\n"
      "str x21, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "252:"  // Height 6: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x14, #0x18\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 253f\n"
      "cmp x14, #0x10\n"
      "mov x9, x12\n"
      "bgt 253f\n"
      "cmp x14, #0x8\n"
      "mov x10, x12\n"
      "bgt 253f\n"
      "mov x11, x12\n"
      "253:"  // Height 6: B setup done
      "cbz x15, 254f\n"
      "ldr q8, [x15, #0x0]\n"
      "ldr q9, [x15, #0x10]\n"
      "ldr q10, [x15, #0x20]\n"
      "ldr q11, [x15, #0x30]\n"
      "add x15, x15, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v28.16b, v8.16b\n"
      "mov v29.16b, v9.16b\n"
      "mov v30.16b, v10.16b\n"
      "mov v31.16b, v11.16b\n"
      "b 273f\n"
      "254:"  // Height 6: no bias
      "tbz %x[flags], #0, 272f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x14, #0x20\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "bge 271f\n"
      "tbz x14, #4, 262f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v20.8h }, [x24], #0x10\n"
      "ld1 { v24.8h }, [x23], #0x10\n"
      "ld1 { v28.8h }, [x22], #0x10\n"
      "ld1 { v9.8h }, [x13], #0x10\n"
      "ld1 { v13.8h }, [x26], #0x10\n"
      "ld1 { v17.8h }, [x25], #0x10\n"
      "ld1 { v21.8h }, [x24], #0x10\n"
      "ld1 { v25.8h }, [x23], #0x10\n"
      "ld1 { v29.8h }, [x22], #0x10\n"
      "tbz x14, #3, 258f\n"
      "ld1 { v10.8h }, [x13], #0x10\n"
      "ld1 { v14.8h }, [x26], #0x10\n"
      "ld1 { v18.8h }, [x25], #0x10\n"
      "ld1 { v22.8h }, [x24], #0x10\n"
      "ld1 { v26.8h }, [x23], #0x10\n"
      "ld1 { v30.8h }, [x22], #0x10\n"
      "tbz x14, #2, 256f\n"
      "ldr d11, [x13], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "ldr d27, [x23], #0x8\n"
      "ldr d31, [x22], #0x8\n"
      "tbz x14, #1, 255f\n"
      "ld1 { v11.s }[2], [x13], #0x4\n"
      "ld1 { v15.s }[2], [x26], #0x4\n"
      "mov x20, #0x3c\n"
      "ld1 { v19.s }[2], [x25], #0x4\n"
      "ld1 { v23.s }[2], [x24], #0x4\n"
      "ld1 { v27.s }[2], [x23], #0x4\n"
      "ld1 { v31.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v11.h }[6], [x13]\n"
      "ld1 { v15.h }[6], [x26]\n"
      "ld1 { v19.h }[6], [x25]\n"
      "ld1 { v23.h }[6], [x24]\n"
      "ld1 { v27.h }[6], [x23]\n"
      "ld1 { v31.h }[6], [x22]\n"
      "b 270f\n"
      "255:"  // Height 6: Partial accumulate: partial_1_28
      "mov x20, #0x38\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v11.h }[4], [x13]\n"
      "ld1 { v15.h }[4], [x26]\n"
      "ld1 { v19.h }[4], [x25]\n"
      "ld1 { v23.h }[4], [x24]\n"
      "ld1 { v27.h }[4], [x23]\n"
      "ld1 { v31.h }[4], [x22]\n"
      "b 270f\n"
      "256:"  // Height 6: Partial accumulate: partial_2_24
      "tbz x14, #1, 257f\n"
      "ldr s11, [x13], #0x4\n"
      "ldr s15, [x26], #0x4\n"
      "mov x20, #0x34\n"
      "ldr s19, [x25], #0x4\n"
      "ldr s23, [x24], #0x4\n"
      "ldr s27, [x23], #0x4\n"
      "ldr s31, [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v11.h }[2], [x13]\n"
      "ld1 { v15.h }[2], [x26]\n"
      "ld1 { v19.h }[2], [x25]\n"
      "ld1 { v23.h }[2], [x24]\n"
      "ld1 { v27.h }[2], [x23]\n"
      "ld1 { v31.h }[2], [x22]\n"
      "b 270f\n"
      "257:"  // Height 6: Partial accumulate: partial_1_24
      "mov x20, #0x30\n"
      "tbz x14, #0, 270f\n"
      "ldr h11, [x13, #0x0]\n"
      "ldr h15, [x26, #0x0]\n"
      "ldr h19, [x25, #0x0]\n"
      "ldr h23, [x24, #0x0]\n"
      "ldr h27, [x23, #0x0]\n"
      "ldr h31, [x22, #0x0]\n"
      "b 270f\n"
      "258:"  // Height 6: Partial accumulate: partial_4_16
      "tbz x14, #2, 260f\n"
      "ldr d10, [x13], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "ldr d26, [x23], #0x8\n"
      "ldr d30, [x22], #0x8\n"
      "tbz x14, #1, 259f\n"
      "ld1 { v10.s }[2], [x13], #0x4\n"
      "ld1 { v14.s }[2], [x26], #0x4\n"
      "mov x20, #0x2c\n"
      "ld1 { v18.s }[2], [x25], #0x4\n"
      "ld1 { v22.s }[2], [x24], #0x4\n"
      "ld1 { v26.s }[2], [x23], #0x4\n"
      "ld1 { v30.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v10.h }[6], [x13]\n"
      "ld1 { v14.h }[6], [x26]\n"
      "ld1 { v18.h }[6], [x25]\n"
      "ld1 { v22.h }[6], [x24]\n"
      "ld1 { v26.h }[6], [x23]\n"
      "ld1 { v30.h }[6], [x22]\n"
      "b 270f\n"
      "259:"  // Height 6: Partial accumulate: partial_1_20
      "mov x20, #0x28\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v10.h }[4], [x13]\n"
      "ld1 { v14.h }[4], [x26]\n"
      "ld1 { v18.h }[4], [x25]\n"
      "ld1 { v22.h }[4], [x24]\n"
      "ld1 { v26.h }[4], [x23]\n"
      "ld1 { v30.h }[4], [x22]\n"
      "b 270f\n"
      "260:"  // Height 6: Partial accumulate: partial_2_16
      "tbz x14, #1, 261f\n"
      "ldr s10, [x13], #0x4\n"
      "ldr s14, [x26], #0x4\n"
      "mov x20, #0x24\n"
      "ldr s18, [x25], #0x4\n"
      "ldr s22, [x24], #0x4\n"
      "ldr s26, [x23], #0x4\n"
      "ldr s30, [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v10.h }[2], [x13]\n"
      "ld1 { v14.h }[2], [x26]\n"
      "ld1 { v18.h }[2], [x25]\n"
      "ld1 { v22.h }[2], [x24]\n"
      "ld1 { v26.h }[2], [x23]\n"
      "ld1 { v30.h }[2], [x22]\n"
      "b 270f\n"
      "261:"  // Height 6: Partial accumulate: partial_1_16
      "mov x20, #0x20\n"
      "tbz x14, #0, 270f\n"
      "ldr h10, [x13, #0x0]\n"
      "ldr h14, [x26, #0x0]\n"
      "ldr h18, [x25, #0x0]\n"
      "ldr h22, [x24, #0x0]\n"
      "ldr h26, [x23, #0x0]\n"
      "ldr h30, [x22, #0x0]\n"
      "b 270f\n"
      "262:"  // Height 6: Partial accumulate: partial_8_0
      "tbz x14, #3, 266f\n"
      "ld1 { v8.8h }, [x13], #0x10\n"
      "ld1 { v12.8h }, [x26], #0x10\n"
      "ld1 { v16.8h }, [x25], #0x10\n"
      "ld1 { v20.8h }, [x24], #0x10\n"
      "ld1 { v24.8h }, [x23], #0x10\n"
      "ld1 { v28.8h }, [x22], #0x10\n"
      "tbz x14, #2, 264f\n"
      "ldr d9, [x13], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "ldr d29, [x22], #0x8\n"
      "tbz x14, #1, 263f\n"
      "ld1 { v9.s }[2], [x13], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "ld1 { v21.s }[2], [x24], #0x4\n"
      "ld1 { v25.s }[2], [x23], #0x4\n"
      "ld1 { v29.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v9.h }[6], [x13]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "ld1 { v21.h }[6], [x24]\n"
      "ld1 { v25.h }[6], [x23]\n"
      "ld1 { v29.h }[6], [x22]\n"
      "b 270f\n"
      "263:"  // Height 6: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v9.h }[4], [x13]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "ld1 { v21.h }[4], [x24]\n"
      "ld1 { v25.h }[4], [x23]\n"
      "ld1 { v29.h }[4], [x22]\n"
      "b 270f\n"
      "264:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x14, #1, 265f\n"
      "ldr s9, [x13], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "ldr s21, [x24], #0x4\n"
      "ldr s25, [x23], #0x4\n"
      "ldr s29, [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v9.h }[2], [x13]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "ld1 { v21.h }[2], [x24]\n"
      "ld1 { v25.h }[2], [x23]\n"
      "ld1 { v29.h }[2], [x22]\n"
      "b 270f\n"
      "265:"  // Height 6: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x14, #0, 270f\n"
      "ldr h9, [x13, #0x0]\n"
      "ldr h13, [x26, #0x0]\n"
      "ldr h17, [x25, #0x0]\n"
      "ldr h21, [x24, #0x0]\n"
      "ldr h25, [x23, #0x0]\n"
      "ldr h29, [x22, #0x0]\n"
      "b 270f\n"
      "266:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x14, #2, 268f\n"
      "ldr d8, [x13], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d28, [x22], #0x8\n"
      "tbz x14, #1, 267f\n"
      "ld1 { v8.s }[2], [x13], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "ld1 { v20.s }[2], [x24], #0x4\n"
      "ld1 { v24.s }[2], [x23], #0x4\n"
      "ld1 { v28.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v8.h }[6], [x13]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "ld1 { v20.h }[6], [x24]\n"
      "ld1 { v24.h }[6], [x23]\n"
      "ld1 { v28.h }[6], [x22]\n"
      "b 270f\n"
      "267:"  // Height 6: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v8.h }[4], [x13]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "ld1 { v20.h }[4], [x24]\n"
      "ld1 { v24.h }[4], [x23]\n"
      "ld1 { v28.h }[4], [x22]\n"
      "b 270f\n"
      "268:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x14, #1, 269f\n"
      "ldr s8, [x13], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "ldr s24, [x23], #0x4\n"
      "ldr s28, [x22], #0x4\n"
      "tbz x14, #0, 270f\n"
      "ld1 { v8.h }[2], [x13]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "ld1 { v20.h }[2], [x24]\n"
      "ld1 { v24.h }[2], [x23]\n"
      "ld1 { v28.h }[2], [x22]\n"
      "b 270f\n"
      "269:"  // Height 6: Partial accumulate: partial_1_0
      "ldr h8, [x13, #0x0]\n"
      "ldr h12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr h16, [x25, #0x0]\n"
      "ldr h20, [x24, #0x0]\n"
      "ldr h24, [x23, #0x0]\n"
      "ldr h28, [x22, #0x0]\n"
      "270:"  // Height 6: Partial accumulate: Done
      "sub x13, x13, x20\n"
      "b 273f\n"
      "271:"  // Height 6: full accumulate
      "ldr q8, [x13, #0x0]\n"
      "ldr q9, [x13, #0x10]\n"
      "ldr q10, [x13, #0x20]\n"
      "ldr q11, [x13, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0x0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "ldr q20, [x24, #0x0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "ldr q24, [x23, #0x0]\n"
      "ldr q25, [x23, #0x10]\n"
      "ldr q26, [x23, #0x20]\n"
      "ldr q27, [x23, #0x30]\n"
      "ldr q28, [x22, #0x0]\n"
      "ldr q29, [x22, #0x10]\n"
      "ldr q30, [x22, #0x20]\n"
      "ldr q31, [x22, #0x30]\n"
      "b 273f\n"
      "272:"  // Height 6: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "273:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "274:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 275f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 276f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "add x21, x21, x20, LSL #1\n"
      "b 276f\n"
      "275:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "add x21, x22, x21, LSL #1\n"
      "276:"  // Height 6: input setup done
      "cmp x27, #0x8\n"
      "blt 279f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q6, [x12, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "blt 278f\n"
      "277:"  // Height 6: Multiply loop: Main loop head
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "fmla v20.8h, v6.8h, v3.h[0]\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v24.8h, v6.8h, v4.h[0]\n"
      "fmla v28.8h, v6.8h, v5.h[0]\n"
      "ldr q6, [x10, #0x0]\n"
      "add x24, x24, #0x10\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "fmla v21.8h, v7.8h, v3.h[0]\n"
      "add x21, x21, #0x10\n"
      "fmla v25.8h, v7.8h, v4.h[0]\n"
      "fmla v29.8h, v7.8h, v5.h[0]\n"
      "ldr q7, [x9, #0x0]\n"
      "fmla v10.8h, v6.8h, v0.h[0]\n"
      "fmla v14.8h, v6.8h, v1.h[0]\n"
      "fmla v18.8h, v6.8h, v2.h[0]\n"
      "fmla v22.8h, v6.8h, v3.h[0]\n"
      "fmla v26.8h, v6.8h, v4.h[0]\n"
      "fmla v30.8h, v6.8h, v5.h[0]\n"
      "ldr q6, [x12, #0x10]\n"
      "fmla v11.8h, v7.8h, v0.h[0]\n"
      "fmla v15.8h, v7.8h, v1.h[0]\n"
      "fmla v19.8h, v7.8h, v2.h[0]\n"
      "fmla v23.8h, v7.8h, v3.h[0]\n"
      "fmla v27.8h, v7.8h, v4.h[0]\n"
      "fmla v31.8h, v7.8h, v5.h[0]\n"
      "ldr q7, [x11, #0x10]\n"
      "fmla v8.8h, v6.8h, v0.h[1]\n"
      "fmla v12.8h, v6.8h, v1.h[1]\n"
      "fmla v16.8h, v6.8h, v2.h[1]\n"
      "fmla v20.8h, v6.8h, v3.h[1]\n"
      "fmla v24.8h, v6.8h, v4.h[1]\n"
      "fmla v28.8h, v6.8h, v5.h[1]\n"
      "ldr q6, [x10, #0x10]\n"
      "fmla v9.8h, v7.8h, v0.h[1]\n"
      "fmla v13.8h, v7.8h, v1.h[1]\n"
      "fmla v17.8h, v7.8h, v2.h[1]\n"
      "fmla v21.8h, v7.8h, v3.h[1]\n"
      "fmla v25.8h, v7.8h, v4.h[1]\n"
      "fmla v29.8h, v7.8h, v5.h[1]\n"
      "ldr q7, [x9, #0x10]\n"
      "fmla v10.8h, v6.8h, v0.h[1]\n"
      "fmla v14.8h, v6.8h, v1.h[1]\n"
      "fmla v18.8h, v6.8h, v2.h[1]\n"
      "fmla v22.8h, v6.8h, v3.h[1]\n"
      "fmla v26.8h, v6.8h, v4.h[1]\n"
      "fmla v30.8h, v6.8h, v5.h[1]\n"
      "ldr q6, [x12, #0x20]\n"
      "fmla v11.8h, v7.8h, v0.h[1]\n"
      "fmla v15.8h, v7.8h, v1.h[1]\n"
      "fmla v19.8h, v7.8h, v2.h[1]\n"
      "fmla v23.8h, v7.8h, v3.h[1]\n"
      "fmla v27.8h, v7.8h, v4.h[1]\n"
      "fmla v31.8h, v7.8h, v5.h[1]\n"
      "ldr q7, [x11, #0x20]\n"
      "fmla v8.8h, v6.8h, v0.h[2]\n"
      "fmla v12.8h, v6.8h, v1.h[2]\n"
      "fmla v16.8h, v6.8h, v2.h[2]\n"
      "fmla v20.8h, v6.8h, v3.h[2]\n"
      "fmla v24.8h, v6.8h, v4.h[2]\n"
      "fmla v28.8h, v6.8h, v5.h[2]\n"
      "ldr q6, [x10, #0x20]\n"
      "fmla v9.8h, v7.8h, v0.h[2]\n"
      "fmla v13.8h, v7.8h, v1.h[2]\n"
      "fmla v17.8h, v7.8h, v2.h[2]\n"
      "fmla v21.8h, v7.8h, v3.h[2]\n"
      "fmla v25.8h, v7.8h, v4.h[2]\n"
      "fmla v29.8h, v7.8h, v5.h[2]\n"
      "ldr q7, [x9, #0x20]\n"
      "fmla v10.8h, v6.8h, v0.h[2]\n"
      "fmla v14.8h, v6.8h, v1.h[2]\n"
      "fmla v18.8h, v6.8h, v2.h[2]\n"
      "fmla v22.8h, v6.8h, v3.h[2]\n"
      "fmla v26.8h, v6.8h, v4.h[2]\n"
      "fmla v30.8h, v6.8h, v5.h[2]\n"
      "ldr q6, [x12, #0x30]\n"
      "fmla v11.8h, v7.8h, v0.h[2]\n"
      "fmla v15.8h, v7.8h, v1.h[2]\n"
      "fmla v19.8h, v7.8h, v2.h[2]\n"
      "fmla v23.8h, v7.8h, v3.h[2]\n"
      "fmla v27.8h, v7.8h, v4.h[2]\n"
      "fmla v31.8h, v7.8h, v5.h[2]\n"
      "ldr q7, [x11, #0x30]\n"
      "fmla v8.8h, v6.8h, v0.h[3]\n"
      "fmla v12.8h, v6.8h, v1.h[3]\n"
      "fmla v16.8h, v6.8h, v2.h[3]\n"
      "fmla v20.8h, v6.8h, v3.h[3]\n"
      "fmla v24.8h, v6.8h, v4.h[3]\n"
      "fmla v28.8h, v6.8h, v5.h[3]\n"
      "ldr q6, [x10, #0x30]\n"
      "fmla v9.8h, v7.8h, v0.h[3]\n"
      "fmla v13.8h, v7.8h, v1.h[3]\n"
      "fmla v17.8h, v7.8h, v2.h[3]\n"
      "fmla v21.8h, v7.8h, v3.h[3]\n"
      "fmla v25.8h, v7.8h, v4.h[3]\n"
      "fmla v29.8h, v7.8h, v5.h[3]\n"
      "ldr q7, [x9, #0x30]\n"
      "fmla v10.8h, v6.8h, v0.h[3]\n"
      "fmla v14.8h, v6.8h, v1.h[3]\n"
      "fmla v18.8h, v6.8h, v2.h[3]\n"
      "fmla v22.8h, v6.8h, v3.h[3]\n"
      "fmla v26.8h, v6.8h, v4.h[3]\n"
      "fmla v30.8h, v6.8h, v5.h[3]\n"
      "ldr q6, [x12, #0x40]\n"
      "fmla v11.8h, v7.8h, v0.h[3]\n"
      "fmla v15.8h, v7.8h, v1.h[3]\n"
      "fmla v19.8h, v7.8h, v2.h[3]\n"
      "fmla v23.8h, v7.8h, v3.h[3]\n"
      "fmla v27.8h, v7.8h, v4.h[3]\n"
      "fmla v31.8h, v7.8h, v5.h[3]\n"
      "ldr q7, [x11, #0x40]\n"
      "fmla v8.8h, v6.8h, v0.h[4]\n"
      "fmla v12.8h, v6.8h, v1.h[4]\n"
      "fmla v16.8h, v6.8h, v2.h[4]\n"
      "fmla v20.8h, v6.8h, v3.h[4]\n"
      "fmla v24.8h, v6.8h, v4.h[4]\n"
      "fmla v28.8h, v6.8h, v5.h[4]\n"
      "ldr q6, [x10, #0x40]\n"
      "fmla v9.8h, v7.8h, v0.h[4]\n"
      "fmla v13.8h, v7.8h, v1.h[4]\n"
      "fmla v17.8h, v7.8h, v2.h[4]\n"
      "fmla v21.8h, v7.8h, v3.h[4]\n"
      "fmla v25.8h, v7.8h, v4.h[4]\n"
      "fmla v29.8h, v7.8h, v5.h[4]\n"
      "ldr q7, [x9, #0x40]\n"
      "fmla v10.8h, v6.8h, v0.h[4]\n"
      "fmla v14.8h, v6.8h, v1.h[4]\n"
      "fmla v18.8h, v6.8h, v2.h[4]\n"
      "fmla v22.8h, v6.8h, v3.h[4]\n"
      "fmla v26.8h, v6.8h, v4.h[4]\n"
      "fmla v30.8h, v6.8h, v5.h[4]\n"
      "ldr q6, [x12, #0x50]\n"
      "fmla v11.8h, v7.8h, v0.h[4]\n"
      "fmla v15.8h, v7.8h, v1.h[4]\n"
      "fmla v19.8h, v7.8h, v2.h[4]\n"
      "fmla v23.8h, v7.8h, v3.h[4]\n"
      "fmla v27.8h, v7.8h, v4.h[4]\n"
      "fmla v31.8h, v7.8h, v5.h[4]\n"
      "ldr q7, [x11, #0x50]\n"
      "fmla v8.8h, v6.8h, v0.h[5]\n"
      "fmla v12.8h, v6.8h, v1.h[5]\n"
      "fmla v16.8h, v6.8h, v2.h[5]\n"
      "fmla v20.8h, v6.8h, v3.h[5]\n"
      "fmla v24.8h, v6.8h, v4.h[5]\n"
      "fmla v28.8h, v6.8h, v5.h[5]\n"
      "ldr q6, [x10, #0x50]\n"
      "fmla v9.8h, v7.8h, v0.h[5]\n"
      "fmla v13.8h, v7.8h, v1.h[5]\n"
      "fmla v17.8h, v7.8h, v2.h[5]\n"
      "fmla v21.8h, v7.8h, v3.h[5]\n"
      "fmla v25.8h, v7.8h, v4.h[5]\n"
      "fmla v29.8h, v7.8h, v5.h[5]\n"
      "ldr q7, [x9, #0x50]\n"
      "fmla v10.8h, v6.8h, v0.h[5]\n"
      "fmla v14.8h, v6.8h, v1.h[5]\n"
      "fmla v18.8h, v6.8h, v2.h[5]\n"
      "fmla v22.8h, v6.8h, v3.h[5]\n"
      "fmla v26.8h, v6.8h, v4.h[5]\n"
      "fmla v30.8h, v6.8h, v5.h[5]\n"
      "ldr q6, [x12, #0x60]\n"
      "fmla v11.8h, v7.8h, v0.h[5]\n"
      "fmla v15.8h, v7.8h, v1.h[5]\n"
      "fmla v19.8h, v7.8h, v2.h[5]\n"
      "fmla v23.8h, v7.8h, v3.h[5]\n"
      "fmla v27.8h, v7.8h, v4.h[5]\n"
      "fmla v31.8h, v7.8h, v5.h[5]\n"
      "ldr q7, [x11, #0x60]\n"
      "fmla v8.8h, v6.8h, v0.h[6]\n"
      "fmla v12.8h, v6.8h, v1.h[6]\n"
      "fmla v16.8h, v6.8h, v2.h[6]\n"
      "fmla v20.8h, v6.8h, v3.h[6]\n"
      "fmla v24.8h, v6.8h, v4.h[6]\n"
      "fmla v28.8h, v6.8h, v5.h[6]\n"
      "ldr q6, [x10, #0x60]\n"
      "fmla v9.8h, v7.8h, v0.h[6]\n"
      "fmla v13.8h, v7.8h, v1.h[6]\n"
      "fmla v17.8h, v7.8h, v2.h[6]\n"
      "fmla v21.8h, v7.8h, v3.h[6]\n"
      "fmla v25.8h, v7.8h, v4.h[6]\n"
      "fmla v29.8h, v7.8h, v5.h[6]\n"
      "ldr q7, [x9, #0x60]\n"
      "fmla v10.8h, v6.8h, v0.h[6]\n"
      "fmla v14.8h, v6.8h, v1.h[6]\n"
      "fmla v18.8h, v6.8h, v2.h[6]\n"
      "fmla v22.8h, v6.8h, v3.h[6]\n"
      "fmla v26.8h, v6.8h, v4.h[6]\n"
      "fmla v30.8h, v6.8h, v5.h[6]\n"
      "ldr q6, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v7.8h, v0.h[6]\n"
      "fmla v15.8h, v7.8h, v1.h[6]\n"
      "fmla v19.8h, v7.8h, v2.h[6]\n"
      "fmla v23.8h, v7.8h, v3.h[6]\n"
      "fmla v27.8h, v7.8h, v4.h[6]\n"
      "fmla v31.8h, v7.8h, v5.h[6]\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v6.8h, v0.h[7]\n"
      "fmla v12.8h, v6.8h, v1.h[7]\n"
      "fmla v16.8h, v6.8h, v2.h[7]\n"
      "fmla v20.8h, v6.8h, v3.h[7]\n"
      "fmla v24.8h, v6.8h, v4.h[7]\n"
      "fmla v28.8h, v6.8h, v5.h[7]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v7.8h, v0.h[7]\n"
      "fmla v13.8h, v7.8h, v1.h[7]\n"
      "fmla v17.8h, v7.8h, v2.h[7]\n"
      "fmla v21.8h, v7.8h, v3.h[7]\n"
      "fmla v25.8h, v7.8h, v4.h[7]\n"
      "fmla v29.8h, v7.8h, v5.h[7]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v6.8h, v0.h[7]\n"
      "fmla v14.8h, v6.8h, v1.h[7]\n"
      "fmla v18.8h, v6.8h, v2.h[7]\n"
      "fmla v22.8h, v6.8h, v3.h[7]\n"
      "fmla v26.8h, v6.8h, v4.h[7]\n"
      "fmla v30.8h, v6.8h, v5.h[7]\n"
      "ldr q6, [x12, #0x0]\n"
      "fmla v11.8h, v7.8h, v0.h[7]\n"
      "ldr q0, [x26, #0x0]\n"
      "fmla v15.8h, v7.8h, v1.h[7]\n"
      "ldr q1, [x25, #0x0]\n"
      "fmla v19.8h, v7.8h, v2.h[7]\n"
      "ldr q2, [x24, #0x0]\n"
      "fmla v23.8h, v7.8h, v3.h[7]\n"
      "ldr q3, [x23, #0x0]\n"
      "fmla v27.8h, v7.8h, v4.h[7]\n"
      "ldr q4, [x22, #0x0]\n"
      "fmla v31.8h, v7.8h, v5.h[7]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q7, [x11, #0x0]\n"
      "bge 277b\n"
      "278:"  // Height 6: Multiply loop: Single iteration only
      "fmla v8.8h, v6.8h, v0.h[0]\n"
      "fmla v12.8h, v6.8h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "fmla v16.8h, v6.8h, v2.h[0]\n"
      "fmla v20.8h, v6.8h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v24.8h, v6.8h, v4.h[0]\n"
      "fmla v28.8h, v6.8h, v5.h[0]\n"
      "ldr q6, [x10, #0x0]\n"
      "add x23, x23, #0x10\n"
      "fmla v9.8h, v7.8h, v0.h[0]\n"
      "fmla v13.8h, v7.8h, v1.h[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      "fmla v17.8h, v7.8h, v2.h[0]\n"
      "fmla v21.8h, v7.8h, v3.h[0]\n"
      "fmla v25.8h, v7.8h, v4.h[0]\n"
      "fmla v29.8h, v7.8h, v5.h[0]\n"
      "ldr q7, [x9, #0x0]\n"
      "fmla v10.8h, v6.8h, v0.h[0]\n"
      "fmla v14.8h, v6.8h, v1.h[0]\n"
      "fmla v18.8h, v6.8h, v2.h[0]\n"
      "fmla v22.8h, v6.8h, v3.h[0]\n"
      "fmla v26.8h, v6.8h, v4.h[0]\n"
      "fmla v30.8h, v6.8h, v5.h[0]\n"
      "ldr q6, [x12, #0x10]\n"
      "fmla v11.8h, v7.8h, v0.h[0]\n"
      "fmla v15.8h, v7.8h, v1.h[0]\n"
      "fmla v19.8h, v7.8h, v2.h[0]\n"
      "fmla v23.8h, v7.8h, v3.h[0]\n"
      "fmla v27.8h, v7.8h, v4.h[0]\n"
      "fmla v31.8h, v7.8h, v5.h[0]\n"
      "ldr q7, [x11, #0x10]\n"
      "fmla v8.8h, v6.8h, v0.h[1]\n"
      "fmla v12.8h, v6.8h, v1.h[1]\n"
      "fmla v16.8h, v6.8h, v2.h[1]\n"
      "fmla v20.8h, v6.8h, v3.h[1]\n"
      "fmla v24.8h, v6.8h, v4.h[1]\n"
      "fmla v28.8h, v6.8h, v5.h[1]\n"
      "ldr q6, [x10, #0x10]\n"
      "fmla v9.8h, v7.8h, v0.h[1]\n"
      "fmla v13.8h, v7.8h, v1.h[1]\n"
      "fmla v17.8h, v7.8h, v2.h[1]\n"
      "fmla v21.8h, v7.8h, v3.h[1]\n"
      "fmla v25.8h, v7.8h, v4.h[1]\n"
      "fmla v29.8h, v7.8h, v5.h[1]\n"
      "ldr q7, [x9, #0x10]\n"
      "fmla v10.8h, v6.8h, v0.h[1]\n"
      "fmla v14.8h, v6.8h, v1.h[1]\n"
      "fmla v18.8h, v6.8h, v2.h[1]\n"
      "fmla v22.8h, v6.8h, v3.h[1]\n"
      "fmla v26.8h, v6.8h, v4.h[1]\n"
      "fmla v30.8h, v6.8h, v5.h[1]\n"
      "ldr q6, [x12, #0x20]\n"
      "fmla v11.8h, v7.8h, v0.h[1]\n"
      "fmla v15.8h, v7.8h, v1.h[1]\n"
      "fmla v19.8h, v7.8h, v2.h[1]\n"
      "fmla v23.8h, v7.8h, v3.h[1]\n"
      "fmla v27.8h, v7.8h, v4.h[1]\n"
      "fmla v31.8h, v7.8h, v5.h[1]\n"
      "ldr q7, [x11, #0x20]\n"
      "fmla v8.8h, v6.8h, v0.h[2]\n"
      "fmla v12.8h, v6.8h, v1.h[2]\n"
      "fmla v16.8h, v6.8h, v2.h[2]\n"
      "fmla v20.8h, v6.8h, v3.h[2]\n"
      "fmla v24.8h, v6.8h, v4.h[2]\n"
      "fmla v28.8h, v6.8h, v5.h[2]\n"
      "ldr q6, [x10, #0x20]\n"
      "fmla v9.8h, v7.8h, v0.h[2]\n"
      "fmla v13.8h, v7.8h, v1.h[2]\n"
      "fmla v17.8h, v7.8h, v2.h[2]\n"
      "fmla v21.8h, v7.8h, v3.h[2]\n"
      "fmla v25.8h, v7.8h, v4.h[2]\n"
      "fmla v29.8h, v7.8h, v5.h[2]\n"
      "ldr q7, [x9, #0x20]\n"
      "fmla v10.8h, v6.8h, v0.h[2]\n"
      "fmla v14.8h, v6.8h, v1.h[2]\n"
      "fmla v18.8h, v6.8h, v2.h[2]\n"
      "fmla v22.8h, v6.8h, v3.h[2]\n"
      "fmla v26.8h, v6.8h, v4.h[2]\n"
      "fmla v30.8h, v6.8h, v5.h[2]\n"
      "ldr q6, [x12, #0x30]\n"
      "fmla v11.8h, v7.8h, v0.h[2]\n"
      "fmla v15.8h, v7.8h, v1.h[2]\n"
      "fmla v19.8h, v7.8h, v2.h[2]\n"
      "fmla v23.8h, v7.8h, v3.h[2]\n"
      "fmla v27.8h, v7.8h, v4.h[2]\n"
      "fmla v31.8h, v7.8h, v5.h[2]\n"
      "ldr q7, [x11, #0x30]\n"
      "fmla v8.8h, v6.8h, v0.h[3]\n"
      "fmla v12.8h, v6.8h, v1.h[3]\n"
      "fmla v16.8h, v6.8h, v2.h[3]\n"
      "fmla v20.8h, v6.8h, v3.h[3]\n"
      "fmla v24.8h, v6.8h, v4.h[3]\n"
      "fmla v28.8h, v6.8h, v5.h[3]\n"
      "ldr q6, [x10, #0x30]\n"
      "fmla v9.8h, v7.8h, v0.h[3]\n"
      "fmla v13.8h, v7.8h, v1.h[3]\n"
      "fmla v17.8h, v7.8h, v2.h[3]\n"
      "fmla v21.8h, v7.8h, v3.h[3]\n"
      "fmla v25.8h, v7.8h, v4.h[3]\n"
      "fmla v29.8h, v7.8h, v5.h[3]\n"
      "ldr q7, [x9, #0x30]\n"
      "fmla v10.8h, v6.8h, v0.h[3]\n"
      "fmla v14.8h, v6.8h, v1.h[3]\n"
      "fmla v18.8h, v6.8h, v2.h[3]\n"
      "fmla v22.8h, v6.8h, v3.h[3]\n"
      "fmla v26.8h, v6.8h, v4.h[3]\n"
      "fmla v30.8h, v6.8h, v5.h[3]\n"
      "ldr q6, [x12, #0x40]\n"
      "fmla v11.8h, v7.8h, v0.h[3]\n"
      "fmla v15.8h, v7.8h, v1.h[3]\n"
      "fmla v19.8h, v7.8h, v2.h[3]\n"
      "fmla v23.8h, v7.8h, v3.h[3]\n"
      "fmla v27.8h, v7.8h, v4.h[3]\n"
      "fmla v31.8h, v7.8h, v5.h[3]\n"
      "ldr q7, [x11, #0x40]\n"
      "fmla v8.8h, v6.8h, v0.h[4]\n"
      "fmla v12.8h, v6.8h, v1.h[4]\n"
      "fmla v16.8h, v6.8h, v2.h[4]\n"
      "fmla v20.8h, v6.8h, v3.h[4]\n"
      "fmla v24.8h, v6.8h, v4.h[4]\n"
      "fmla v28.8h, v6.8h, v5.h[4]\n"
      "ldr q6, [x10, #0x40]\n"
      "fmla v9.8h, v7.8h, v0.h[4]\n"
      "fmla v13.8h, v7.8h, v1.h[4]\n"
      "fmla v17.8h, v7.8h, v2.h[4]\n"
      "fmla v21.8h, v7.8h, v3.h[4]\n"
      "fmla v25.8h, v7.8h, v4.h[4]\n"
      "fmla v29.8h, v7.8h, v5.h[4]\n"
      "ldr q7, [x9, #0x40]\n"
      "fmla v10.8h, v6.8h, v0.h[4]\n"
      "fmla v14.8h, v6.8h, v1.h[4]\n"
      "fmla v18.8h, v6.8h, v2.h[4]\n"
      "fmla v22.8h, v6.8h, v3.h[4]\n"
      "fmla v26.8h, v6.8h, v4.h[4]\n"
      "fmla v30.8h, v6.8h, v5.h[4]\n"
      "ldr q6, [x12, #0x50]\n"
      "fmla v11.8h, v7.8h, v0.h[4]\n"
      "fmla v15.8h, v7.8h, v1.h[4]\n"
      "fmla v19.8h, v7.8h, v2.h[4]\n"
      "fmla v23.8h, v7.8h, v3.h[4]\n"
      "fmla v27.8h, v7.8h, v4.h[4]\n"
      "fmla v31.8h, v7.8h, v5.h[4]\n"
      "ldr q7, [x11, #0x50]\n"
      "fmla v8.8h, v6.8h, v0.h[5]\n"
      "fmla v12.8h, v6.8h, v1.h[5]\n"
      "fmla v16.8h, v6.8h, v2.h[5]\n"
      "fmla v20.8h, v6.8h, v3.h[5]\n"
      "fmla v24.8h, v6.8h, v4.h[5]\n"
      "fmla v28.8h, v6.8h, v5.h[5]\n"
      "ldr q6, [x10, #0x50]\n"
      "fmla v9.8h, v7.8h, v0.h[5]\n"
      "fmla v13.8h, v7.8h, v1.h[5]\n"
      "fmla v17.8h, v7.8h, v2.h[5]\n"
      "fmla v21.8h, v7.8h, v3.h[5]\n"
      "fmla v25.8h, v7.8h, v4.h[5]\n"
      "fmla v29.8h, v7.8h, v5.h[5]\n"
      "ldr q7, [x9, #0x50]\n"
      "fmla v10.8h, v6.8h, v0.h[5]\n"
      "fmla v14.8h, v6.8h, v1.h[5]\n"
      "fmla v18.8h, v6.8h, v2.h[5]\n"
      "fmla v22.8h, v6.8h, v3.h[5]\n"
      "fmla v26.8h, v6.8h, v4.h[5]\n"
      "fmla v30.8h, v6.8h, v5.h[5]\n"
      "ldr q6, [x12, #0x60]\n"
      "fmla v11.8h, v7.8h, v0.h[5]\n"
      "fmla v15.8h, v7.8h, v1.h[5]\n"
      "fmla v19.8h, v7.8h, v2.h[5]\n"
      "fmla v23.8h, v7.8h, v3.h[5]\n"
      "fmla v27.8h, v7.8h, v4.h[5]\n"
      "fmla v31.8h, v7.8h, v5.h[5]\n"
      "ldr q7, [x11, #0x60]\n"
      "fmla v8.8h, v6.8h, v0.h[6]\n"
      "fmla v12.8h, v6.8h, v1.h[6]\n"
      "fmla v16.8h, v6.8h, v2.h[6]\n"
      "fmla v20.8h, v6.8h, v3.h[6]\n"
      "fmla v24.8h, v6.8h, v4.h[6]\n"
      "fmla v28.8h, v6.8h, v5.h[6]\n"
      "ldr q6, [x10, #0x60]\n"
      "fmla v9.8h, v7.8h, v0.h[6]\n"
      "fmla v13.8h, v7.8h, v1.h[6]\n"
      "fmla v17.8h, v7.8h, v2.h[6]\n"
      "fmla v21.8h, v7.8h, v3.h[6]\n"
      "fmla v25.8h, v7.8h, v4.h[6]\n"
      "fmla v29.8h, v7.8h, v5.h[6]\n"
      "ldr q7, [x9, #0x60]\n"
      "fmla v10.8h, v6.8h, v0.h[6]\n"
      "fmla v14.8h, v6.8h, v1.h[6]\n"
      "fmla v18.8h, v6.8h, v2.h[6]\n"
      "fmla v22.8h, v6.8h, v3.h[6]\n"
      "fmla v26.8h, v6.8h, v4.h[6]\n"
      "fmla v30.8h, v6.8h, v5.h[6]\n"
      "ldr q6, [x12, #0x70]\n"
      "add x12, x12, #0x80\n"
      "fmla v11.8h, v7.8h, v0.h[6]\n"
      "fmla v15.8h, v7.8h, v1.h[6]\n"
      "fmla v19.8h, v7.8h, v2.h[6]\n"
      "fmla v23.8h, v7.8h, v3.h[6]\n"
      "fmla v27.8h, v7.8h, v4.h[6]\n"
      "fmla v31.8h, v7.8h, v5.h[6]\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      "fmla v8.8h, v6.8h, v0.h[7]\n"
      "fmla v12.8h, v6.8h, v1.h[7]\n"
      "fmla v16.8h, v6.8h, v2.h[7]\n"
      "fmla v20.8h, v6.8h, v3.h[7]\n"
      "fmla v24.8h, v6.8h, v4.h[7]\n"
      "fmla v28.8h, v6.8h, v5.h[7]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      "fmla v9.8h, v7.8h, v0.h[7]\n"
      "fmla v13.8h, v7.8h, v1.h[7]\n"
      "fmla v17.8h, v7.8h, v2.h[7]\n"
      "fmla v21.8h, v7.8h, v3.h[7]\n"
      "fmla v25.8h, v7.8h, v4.h[7]\n"
      "fmla v29.8h, v7.8h, v5.h[7]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      "fmla v10.8h, v6.8h, v0.h[7]\n"
      "fmla v14.8h, v6.8h, v1.h[7]\n"
      "fmla v18.8h, v6.8h, v2.h[7]\n"
      "fmla v22.8h, v6.8h, v3.h[7]\n"
      "fmla v26.8h, v6.8h, v4.h[7]\n"
      "fmla v30.8h, v6.8h, v5.h[7]\n"
      "fmla v11.8h, v7.8h, v0.h[7]\n"
      "fmla v15.8h, v7.8h, v1.h[7]\n"
      "fmla v19.8h, v7.8h, v2.h[7]\n"
      "fmla v23.8h, v7.8h, v3.h[7]\n"
      "fmla v27.8h, v7.8h, v4.h[7]\n"
      "fmla v31.8h, v7.8h, v5.h[7]\n"
      "279:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 281f\n"
      "280:"  // Height 6: Multiply loop: Odd block loop
      "ldr h7, [x26], #0x2\n"
      "ldr h6, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h5, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h3, [x22], #0x2\n"
      "ldr h2, [x21], #0x2\n"
      "ldr q1, [x12, #0x0]\n"
      "ldr q0, [x11, #0x0]\n"
      "add x12, x12, #0x10\n"
      "add x11, x11, #0x10\n"
      "fmla v8.8h, v1.8h, v7.h[0]\n"
      "fmla v12.8h, v1.8h, v6.h[0]\n"
      "fmla v16.8h, v1.8h, v5.h[0]\n"
      "fmla v20.8h, v1.8h, v4.h[0]\n"
      "fmla v24.8h, v1.8h, v3.h[0]\n"
      "fmla v28.8h, v1.8h, v2.h[0]\n"
      "ldr q1, [x10, #0x0]\n"
      "add x10, x10, #0x10\n"
      "fmla v9.8h, v0.8h, v7.h[0]\n"
      "fmla v13.8h, v0.8h, v6.h[0]\n"
      "fmla v17.8h, v0.8h, v5.h[0]\n"
      "fmla v21.8h, v0.8h, v4.h[0]\n"
      "fmla v25.8h, v0.8h, v3.h[0]\n"
      "fmla v29.8h, v0.8h, v2.h[0]\n"
      "ldr q0, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "fmla v10.8h, v1.8h, v7.h[0]\n"
      "fmla v14.8h, v1.8h, v6.h[0]\n"
      "fmla v18.8h, v1.8h, v5.h[0]\n"
      "fmla v22.8h, v1.8h, v4.h[0]\n"
      "fmla v26.8h, v1.8h, v3.h[0]\n"
      "fmla v30.8h, v1.8h, v2.h[0]\n"
      "fmla v11.8h, v0.8h, v7.h[0]\n"
      "fmla v15.8h, v0.8h, v6.h[0]\n"
      "fmla v19.8h, v0.8h, v5.h[0]\n"
      "fmla v23.8h, v0.8h, v4.h[0]\n"
      "fmla v27.8h, v0.8h, v3.h[0]\n"
      "fmla v31.8h, v0.8h, v2.h[0]\n"
      "cbnz x27, 280b\n"
      "281:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 274b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "tbz %x[flags], #1, 282f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v1.8h }, [x21]\n"
      "ld1r { v0.8h }, [x20]\n"
      "fmin v8.8h, v8.8h, v1.8h\n"
      "fmin v9.8h, v9.8h, v1.8h\n"
      "fmin v10.8h, v10.8h, v1.8h\n"
      "fmin v11.8h, v11.8h, v1.8h\n"
      "fmin v12.8h, v12.8h, v1.8h\n"
      "fmin v13.8h, v13.8h, v1.8h\n"
      "fmin v14.8h, v14.8h, v1.8h\n"
      "fmin v15.8h, v15.8h, v1.8h\n"
      "fmin v16.8h, v16.8h, v1.8h\n"
      "fmin v17.8h, v17.8h, v1.8h\n"
      "fmin v18.8h, v18.8h, v1.8h\n"
      "fmin v19.8h, v19.8h, v1.8h\n"
      "fmin v20.8h, v20.8h, v1.8h\n"
      "fmin v21.8h, v21.8h, v1.8h\n"
      "fmin v22.8h, v22.8h, v1.8h\n"
      "fmin v23.8h, v23.8h, v1.8h\n"
      "fmin v24.8h, v24.8h, v1.8h\n"
      "fmin v25.8h, v25.8h, v1.8h\n"
      "fmin v26.8h, v26.8h, v1.8h\n"
      "fmin v27.8h, v27.8h, v1.8h\n"
      "fmin v28.8h, v28.8h, v1.8h\n"
      "fmin v29.8h, v29.8h, v1.8h\n"
      "fmin v30.8h, v30.8h, v1.8h\n"
      "fmin v31.8h, v31.8h, v1.8h\n"
      "fmax v8.8h, v8.8h, v0.8h\n"
      "fmax v9.8h, v9.8h, v0.8h\n"
      "fmax v10.8h, v10.8h, v0.8h\n"
      "fmax v11.8h, v11.8h, v0.8h\n"
      "fmax v12.8h, v12.8h, v0.8h\n"
      "fmax v13.8h, v13.8h, v0.8h\n"
      "fmax v14.8h, v14.8h, v0.8h\n"
      "fmax v15.8h, v15.8h, v0.8h\n"
      "fmax v16.8h, v16.8h, v0.8h\n"
      "fmax v17.8h, v17.8h, v0.8h\n"
      "fmax v18.8h, v18.8h, v0.8h\n"
      "fmax v19.8h, v19.8h, v0.8h\n"
      "fmax v20.8h, v20.8h, v0.8h\n"
      "fmax v21.8h, v21.8h, v0.8h\n"
      "fmax v22.8h, v22.8h, v0.8h\n"
      "fmax v23.8h, v23.8h, v0.8h\n"
      "fmax v24.8h, v24.8h, v0.8h\n"
      "fmax v25.8h, v25.8h, v0.8h\n"
      "fmax v26.8h, v26.8h, v0.8h\n"
      "fmax v27.8h, v27.8h, v0.8h\n"
      "fmax v28.8h, v28.8h, v0.8h\n"
      "fmax v29.8h, v29.8h, v0.8h\n"
      "fmax v30.8h, v30.8h, v0.8h\n"
      "fmax v31.8h, v31.8h, v0.8h\n"
      "282:"  // Height 6: No activation
      "cmp x14, #0x20\n"
      "bge 299f\n"
      "tbz x14, #4, 290f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v9.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v13.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v17.8h }, [x25], #0x10\n"
      "st1 { v20.8h }, [x24], #0x10\n"
      "st1 { v21.8h }, [x24], #0x10\n"
      "st1 { v24.8h }, [x23], #0x10\n"
      "st1 { v25.8h }, [x23], #0x10\n"
      "st1 { v28.8h }, [x22], #0x10\n"
      "st1 { v29.8h }, [x22], #0x10\n"
      "tbz x14, #3, 286f\n"
      "st1 { v10.8h }, [x13], #0x10\n"
      "st1 { v14.8h }, [x26], #0x10\n"
      "st1 { v18.8h }, [x25], #0x10\n"
      "st1 { v22.8h }, [x24], #0x10\n"
      "st1 { v26.8h }, [x23], #0x10\n"
      "st1 { v30.8h }, [x22], #0x10\n"
      "tbz x14, #2, 284f\n"
      "str d11, [x13], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "str d27, [x23], #0x8\n"
      "str d31, [x22], #0x8\n"
      "tbz x14, #1, 283f\n"
      "st1 { v11.s }[2], [x13], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v19.s }[2], [x25], #0x4\n"
      "st1 { v23.s }[2], [x24], #0x4\n"
      "st1 { v27.s }[2], [x23], #0x4\n"
      "st1 { v31.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v11.h }[6], [x13]\n"
      "st1 { v15.h }[6], [x26]\n"
      "st1 { v19.h }[6], [x25]\n"
      "st1 { v23.h }[6], [x24]\n"
      "st1 { v27.h }[6], [x23]\n"
      "st1 { v31.h }[6], [x22]\n"
      "b 298f\n"
      "283:"  // Height 6: Partial direct writeback: partial_1_28
      "tbz x14, #0, 298f\n"
      "st1 { v11.h }[4], [x13]\n"
      "st1 { v15.h }[4], [x26]\n"
      "st1 { v19.h }[4], [x25]\n"
      "st1 { v23.h }[4], [x24]\n"
      "st1 { v27.h }[4], [x23]\n"
      "st1 { v31.h }[4], [x22]\n"
      "b 298f\n"
      "284:"  // Height 6: Partial direct writeback: partial_2_24
      "tbz x14, #1, 285f\n"
      "str s11, [x13], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s19, [x25], #0x4\n"
      "str s23, [x24], #0x4\n"
      "str s27, [x23], #0x4\n"
      "str s31, [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v11.h }[2], [x13]\n"
      "st1 { v15.h }[2], [x26]\n"
      "st1 { v19.h }[2], [x25]\n"
      "st1 { v23.h }[2], [x24]\n"
      "st1 { v27.h }[2], [x23]\n"
      "st1 { v31.h }[2], [x22]\n"
      "b 298f\n"
      "285:"  // Height 6: Partial direct writeback: partial_1_24
      "tbz x14, #0, 298f\n"
      "str h11, [x13, #0x0]\n"
      "str h15, [x26, #0x0]\n"
      "str h19, [x25, #0x0]\n"
      "str h23, [x24, #0x0]\n"
      "str h27, [x23, #0x0]\n"
      "str h31, [x22, #0x0]\n"
      "b 298f\n"
      "286:"  // Height 6: Partial direct writeback: partial_4_16
      "tbz x14, #2, 288f\n"
      "str d10, [x13], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "str d22, [x24], #0x8\n"
      "str d26, [x23], #0x8\n"
      "str d30, [x22], #0x8\n"
      "tbz x14, #1, 287f\n"
      "st1 { v10.s }[2], [x13], #0x4\n"
      "st1 { v14.s }[2], [x26], #0x4\n"
      "st1 { v18.s }[2], [x25], #0x4\n"
      "st1 { v22.s }[2], [x24], #0x4\n"
      "st1 { v26.s }[2], [x23], #0x4\n"
      "st1 { v30.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v10.h }[6], [x13]\n"
      "st1 { v14.h }[6], [x26]\n"
      "st1 { v18.h }[6], [x25]\n"
      "st1 { v22.h }[6], [x24]\n"
      "st1 { v26.h }[6], [x23]\n"
      "st1 { v30.h }[6], [x22]\n"
      "b 298f\n"
      "287:"  // Height 6: Partial direct writeback: partial_1_20
      "tbz x14, #0, 298f\n"
      "st1 { v10.h }[4], [x13]\n"
      "st1 { v14.h }[4], [x26]\n"
      "st1 { v18.h }[4], [x25]\n"
      "st1 { v22.h }[4], [x24]\n"
      "st1 { v26.h }[4], [x23]\n"
      "st1 { v30.h }[4], [x22]\n"
      "b 298f\n"
      "288:"  // Height 6: Partial direct writeback: partial_2_16
      "tbz x14, #1, 289f\n"
      "str s10, [x13], #0x4\n"
      "str s14, [x26], #0x4\n"
      "str s18, [x25], #0x4\n"
      "str s22, [x24], #0x4\n"
      "str s26, [x23], #0x4\n"
      "str s30, [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v10.h }[2], [x13]\n"
      "st1 { v14.h }[2], [x26]\n"
      "st1 { v18.h }[2], [x25]\n"
      "st1 { v22.h }[2], [x24]\n"
      "st1 { v26.h }[2], [x23]\n"
      "st1 { v30.h }[2], [x22]\n"
      "b 298f\n"
      "289:"  // Height 6: Partial direct writeback: partial_1_16
      "tbz x14, #0, 298f\n"
      "str h10, [x13, #0x0]\n"
      "str h14, [x26, #0x0]\n"
      "str h18, [x25, #0x0]\n"
      "str h22, [x24, #0x0]\n"
      "str h26, [x23, #0x0]\n"
      "str h30, [x22, #0x0]\n"
      "b 298f\n"
      "290:"  // Height 6: Partial direct writeback: partial_8_0
      "tbz x14, #3, 294f\n"
      "st1 { v8.8h }, [x13], #0x10\n"
      "st1 { v12.8h }, [x26], #0x10\n"
      "st1 { v16.8h }, [x25], #0x10\n"
      "st1 { v20.8h }, [x24], #0x10\n"
      "st1 { v24.8h }, [x23], #0x10\n"
      "st1 { v28.8h }, [x22], #0x10\n"
      "tbz x14, #2, 292f\n"
      "str d9, [x13], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "str d29, [x22], #0x8\n"
      "tbz x14, #1, 291f\n"
      "st1 { v9.s }[2], [x13], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "st1 { v21.s }[2], [x24], #0x4\n"
      "st1 { v25.s }[2], [x23], #0x4\n"
      "st1 { v29.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v9.h }[6], [x13]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "st1 { v21.h }[6], [x24]\n"
      "st1 { v25.h }[6], [x23]\n"
      "st1 { v29.h }[6], [x22]\n"
      "b 298f\n"
      "291:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x14, #0, 298f\n"
      "st1 { v9.h }[4], [x13]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "st1 { v21.h }[4], [x24]\n"
      "st1 { v25.h }[4], [x23]\n"
      "st1 { v29.h }[4], [x22]\n"
      "b 298f\n"
      "292:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x14, #1, 293f\n"
      "str s9, [x13], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "str s21, [x24], #0x4\n"
      "str s25, [x23], #0x4\n"
      "str s29, [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v9.h }[2], [x13]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "st1 { v21.h }[2], [x24]\n"
      "st1 { v25.h }[2], [x23]\n"
      "st1 { v29.h }[2], [x22]\n"
      "b 298f\n"
      "293:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x14, #0, 298f\n"
      "str h9, [x13, #0x0]\n"
      "str h13, [x26, #0x0]\n"
      "str h17, [x25, #0x0]\n"
      "str h21, [x24, #0x0]\n"
      "str h25, [x23, #0x0]\n"
      "str h29, [x22, #0x0]\n"
      "b 298f\n"
      "294:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x14, #2, 296f\n"
      "str d8, [x13], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "str d28, [x22], #0x8\n"
      "tbz x14, #1, 295f\n"
      "st1 { v8.s }[2], [x13], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "st1 { v28.s }[2], [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v8.h }[6], [x13]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "st1 { v20.h }[6], [x24]\n"
      "st1 { v24.h }[6], [x23]\n"
      "st1 { v28.h }[6], [x22]\n"
      "b 298f\n"
      "295:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x14, #0, 298f\n"
      "st1 { v8.h }[4], [x13]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "st1 { v20.h }[4], [x24]\n"
      "st1 { v24.h }[4], [x23]\n"
      "st1 { v28.h }[4], [x22]\n"
      "b 298f\n"
      "296:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x14, #1, 297f\n"
      "str s8, [x13], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "str s28, [x22], #0x4\n"
      "tbz x14, #0, 298f\n"
      "st1 { v8.h }[2], [x13]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "st1 { v20.h }[2], [x24]\n"
      "st1 { v24.h }[2], [x23]\n"
      "st1 { v28.h }[2], [x22]\n"
      "b 298f\n"
      "297:"  // Height 6: Partial direct writeback: partial_1_0
      "str h8, [x13, #0x0]\n"
      "str h12, [x26, #0x0]\n"
      "str h16, [x25, #0x0]\n"
      "str h20, [x24, #0x0]\n"
      "str h24, [x23, #0x0]\n"
      "str h28, [x22, #0x0]\n"
      "298:"  // Height 6: Partial direct writeback: Done
      "b 300f\n"
      "299:"  // Height 6: Full writeback
      "str q8, [x13, #0x0]\n"
      "str q9, [x13, #0x10]\n"
      "str q10, [x13, #0x20]\n"
      "str q11, [x13, #0x30]\n"
      "add x13, x13, #0x40\n"
      "str q12, [x26, #0x0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0x0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "str q20, [x24, #0x0]\n"
      "str q21, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "str q23, [x24, #0x30]\n"
      "str q24, [x23, #0x0]\n"
      "str q25, [x23, #0x10]\n"
      "str q26, [x23, #0x20]\n"
      "str q27, [x23, #0x30]\n"
      "str q28, [x22, #0x0]\n"
      "str q29, [x22, #0x10]\n"
      "str q30, [x22, #0x20]\n"
      "str q31, [x22, #0x30]\n"
      "300:"  // Height 6: Writeback done
      "subs x14, x14, #0x20\n"
      "bgt 252b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 302f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 301f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "301:"  // Update direct input
      "mov x20, #0xc\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "302:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_B_stride] "I" (offsetof(KernelArgs, B_stride)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_cur_B_ptr] "I" (offsetof(KernelArgs, cur_B_ptr)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
